(ns vincit.dbwalk.database.sql
  "An implementation for the protocols and multimethods required to use an SQL database as a datasource."
  (:require [clojure.java.jdbc :as jdbc]
            [honeysql.helpers :as hsql]
            [honeysql.core :as sql]
            [vincit.dbwalk.relations :as relations]
            [vincit.dbwalk.query-parser :as parser]
            [vincit.dbwalk.utils :as utils]
            [vincit.dbwalk.action.operations :refer [apply-operation!]]))


(defrecord ColumnSpec [table column]
  relations/RelationalData
  (property-name [this]
    column)
  (table [this]
    table))

(defn- create-where-clause [this source-items]
  (hsql/where [:in (relations/target-property this) (map (relations/source-property this) source-items)]))

(defn- add-merge-clauses-for-relations "Generates the HoneySQL query for SELECTing the next level in the query result."
  [query-node joins-to-follow nodes-to-advance]
  (->> (map #(create-where-clause % nodes-to-advance) joins-to-follow)
       (mapv :where)
       (into [:or])
       (apply hsql/merge-where (parser/query query-node))))

(defn- add-select-clause-for-properties [query required-properties]
  (apply hsql/select query (seq required-properties)))

(defmethod relations/advance-query-tree :dbwalk/jdbc-sql [config query-node relations-to-follow nodes-to-advance properties-to-get]
  (let [data-source (relations/data-source-from-relations relations-to-follow)
        data-source-config (relations/select-data-source-from-configuration config data-source)
        query (-> query-node
                  (add-merge-clauses-for-relations relations-to-follow nodes-to-advance)
                  (add-select-clause-for-properties properties-to-get)
                  (#(relations/pre-query-filter config data-source % nodes-to-advance relations-to-follow))
                  (sql/format))]
    (jdbc/query data-source-config query)))

(defmethod relations/read-from-table :dbwalk/jdbc-sql [configuration data-source query properties-to-select relations-to-follow]
  (let [query (apply hsql/select query (seq properties-to-select))
        data-source-config (relations/select-data-source-from-configuration configuration data-source)
        filtered-query (relations/pre-query-filter configuration data-source query [] relations-to-follow)]
    (jdbc/query data-source-config (sql/format filtered-query))))



;; Action graph operations

(defn- extract-params [configuration data-source-description table entity]
  (let [db-spec (get-in configuration [:dbwalk/data-sources (:data-source data-source-description)])
        primary-key-column (utils/primary-key-column configuration data-source-description table)
        where-clause [(str (name primary-key-column) " = ?") (get entity primary-key-column)]]
    {:db-spec      db-spec
     :where-clause where-clause}))

;; This multimethod exists to avoid manually dispatching via cond.
(defmulti apply-to-jdbc-database! (fn [configuration data-source-description operation table entity]
                                   operation))

(defmethod apply-to-jdbc-database! :insert [configuration data-source-description operation table entity]
  (let [db-spec (get-in configuration [:dbwalk/data-sources (:data-source data-source-description)])]
    (first (jdbc/insert! db-spec table entity))))

(defmethod apply-to-jdbc-database! :delete [configuration data-source-description operation table entity]
  (let [{:keys [db-spec where-clause]} (extract-params configuration data-source-description table entity)]
    (jdbc/delete! db-spec table where-clause)))

(defmethod apply-to-jdbc-database! :update [configuration data-source-description operation table entity]
  (let [{:keys [db-spec where-clause]} (extract-params configuration data-source-description table entity)]
    (first (jdbc/update! db-spec table entity where-clause))))

(defmethod apply-to-jdbc-database! :no-op [configuration data-source-description operation table entity]
  entity)

(defmethod apply-operation! :dbwalk/jdbc-sql [configuration data-source-description operation table entity]
  (apply-to-jdbc-database! configuration data-source-description operation table entity))
