(ns vincit.dbwalk.action.full-map
  "Functions for transforming a nested map in the 'map' output format to a Loom graph which
   can be given to v.d.action.writer. The desired operation (see .operations) is given as metadata.
   See .modify for helper functions.

   The nodes in the graph actually contain the full original subtree from the input map. This causes
   only one node to be created for equivalent subtrees in the entire input tree. The effect is a kind of
   duplicate removal."
  (:require [vincit.dbwalk.relations :as relations]
            [vincit.dbwalk.database.sql]
            [vincit.dbwalk.graph :as graph]
            [vincit.dbwalk.utils :as utils]
            [vincit.dbwalk.action.operations :refer [all-operations]]))

(defn- compare-sources [relation to-namespace]
  (= (relations/source-table relation) to-namespace))

(defn- joins-from-table [configuration data-source-description table]
  (let [relations-from-datasource (get-in configuration [:dbwalk/relations data-source-description])]
    (->> relations-from-datasource
         (filter #(compare-sources % table)))))

(defn- joinable-properties-from-table
  "Names of foreign or primary key columns"
  [config data-source-description table]
  (map relations/created-attribute-name (joins-from-table config data-source-description table)))

(defn- property-to-relation [config data-source-description table]
  (group-by relations/created-attribute-name (joins-from-table config data-source-description table)))

(defn- detachable? [value]
  (or (map? value)
      (sequential? value)))

(defn- properties-with-detachable-data [entity joinable-properties]
  (let [joinables (apply hash-set joinable-properties)]
    (->> (keys entity)
         (filter #(contains? joinables %))
         (filter #(detachable? (get entity %))))))

(defn- select-properties-to-detach [configuration data-source-description table data]
  (let [joinable-properties (joinable-properties-from-table configuration data-source-description table)]
    (->> data
         (map #(properties-with-detachable-data % joinable-properties))
         (flatten)
         (apply hash-set))))

(defn select-operation [entity primary-key-column default-operation]
  (let [selected-ops (-> (meta entity)
                         (select-keys all-operations))]
    (when (> (count selected-ops) 1)
      (throw (ex-info "Too many operations" {:entity entity :operations selected-ops})))
    (cond
      (nil? (get entity primary-key-column)) :insert
      (= 0 (count selected-ops)) default-operation
      :else (first (keys selected-ops)))))

(defn build-data-graph [graph configuration data-source-description default-operation table rows]
  (let [properties (select-properties-to-detach configuration data-source-description table rows)
        relation-map (property-to-relation configuration data-source-description table)
        primary-key-column (utils/primary-key-column configuration data-source-description table)]
    (doseq [entity rows]
      (do
        (graph/add-node! graph entity data-source-description table)
        (graph/set-operation graph entity (select-operation entity primary-key-column default-operation))
        (doseq [prop properties]
          (let [children (get entity prop [])
                join (first (get relation-map prop))]
            (build-data-graph graph configuration (relations/to-data-source join) default-operation (relations/target-table join) children)
            (doseq [child children]
              (if (relations/insert-source-first join)      ;; Link entities in insertion order
                (graph/add-link graph entity child join)
                (graph/add-link graph child entity join)))))
        (graph/update-insert-data! graph entity #(apply dissoc % properties)))))) ;; Remove properties that represented data in other tables
