(ns vincit.dbwalk.writer
  "Utility for writing a deep map to the database. Accepts the default output as its input.
   The input map is separated to a Loom digraph so that the edges between nodes are towards
   the node with the foreign key. This allows correct insertion ordering by always inserting
   nodes (rows) with no incoming edges, pushing the generated primary key to its children
   and repeating until the graph is empty.

   This namespace is for writing tests only, the API will change before it stabilizes."

  (:require [vincit.dbwalk.relations :as relations]
            [vincit.dbwalk.database.sql]
            [vincit.dbwalk.graph :as graph]
            [vincit.dbwalk.config :as config]
            [vincit.dbwalk.schema-detect :as detect]
            [vincit.dbwalk.utils :as utils]))

(defn- compare-sources [relation to-namespace]
  (= (relations/source-table relation) to-namespace))

(defn- joins-from-table [configuration data-source-description table]
  (let [relations-from-datasource (get-in configuration [:dbwalk/relations data-source-description])]
    (->> relations-from-datasource
         (filter #(compare-sources % table)))))

(defn- joinable-properties-from-table [config data-source-description table]
  (map relations/created-attribute-name (joins-from-table config data-source-description table)))

(defn- property-to-relation [config data-source-description table]
  (group-by relations/created-attribute-name (joins-from-table config data-source-description table)))

(defn- detachable? [value]
  (or (map? value)
      (sequential? value)))

(defn- properties-with-detachable-data [entity joinable-properties]
  (let [joinables (apply hash-set joinable-properties)]
    (->> (keys entity)
         (filter #(contains? joinables %))
         (filter #(detachable? (get entity %))))))

(defn- select-properties-to-detach [configuration data-source-description table data]
  (let [joinable-properties (joinable-properties-from-table configuration data-source-description table)]
    (->> data
         (map #(properties-with-detachable-data % joinable-properties))
         (flatten)
         (apply hash-set))))

(defn- build-data-graph [graph configuration data-source-description table data-to-insert]
  (let [properties (select-properties-to-detach configuration data-source-description table data-to-insert)
        relation-map (property-to-relation configuration data-source-description table)]
    (doseq [entity data-to-insert]
      (do
        (graph/add-node! graph entity data-source-description table)
        (doseq [prop properties]
          (let [children (get entity prop [])
                join (first (get relation-map prop))]
            (build-data-graph graph configuration (relations/to-data-source join) (relations/target-table join) children)
            (doseq [child children]
              (if (relations/insert-source-first join)      ;; Link entities in insertion order
                (graph/add-link graph entity child join)
                (graph/add-link graph child entity join)))))
        (graph/update-insert-data! graph entity #(apply dissoc % properties))))))

(defn insert-single-entity [storage configuration entity]
  (let [{:keys [data-source table insert-data]} (graph/get-data-for-node storage entity)
        primary-key (relations/insert-to-table configuration data-source table insert-data)]
    (when primary-key
      (graph/update-key-to-children! storage entity primary-key))
    (graph/remove-node storage entity)))

(defn- insert-data [storage configuration]
  (when-let [first-level-data (seq (graph/root-items storage))]
    (run! #(insert-single-entity storage configuration %) first-level-data)
    (recur storage configuration)))

(defn insert! [configuration data-to-insert]
  (let [{:keys [table data data-source-description]} data-to-insert
        storage (graph/new-graph)]
    (build-data-graph storage configuration data-source-description table data)
    (insert-data storage configuration)))


(defn insert-to-first-datasource!
  "Convenience function for writing tests. Inserts the given data to the first datasource found in configuration.
  Will not work if the configuration contains more than one datasource."
  [configuration data-to-insert]
  (let [first-ds (first (keys (:dbwalk/relations configuration)))]
    (insert! configuration (assoc data-to-insert :data-source-description first-ds))))
