(ns vincit.dbwalk.api.sweet
  "Example functions to give an idea of using this library in an application."
  (:require [vincit.dbwalk.graph :as graph]
            [vincit.dbwalk.utils :as utils]
            [vincit.dbwalk.schema-detect :as detect]
            [vincit.dbwalk.config :as config]
            [vincit.dbwalk.crawler :as crawler]
            [vincit.dbwalk.output.flat :as flat]
            [vincit.dbwalk.output.filtered :as filtered]
            [vincit.dbwalk.output.map :as out-map]
            [vincit.dbwalk.api.subgraph :as subgraph]
            [vincit.dbwalk.api.tree-map :as tree-map]
            [schema.core :as s]))

(defn query-single-db
  "Runs the given query. Returns a Loom graph."
  [db-spec query-tree]
  (let [datasource {:data-source :db
                    :type        :dbwalk/source-sql}
        config (merge (config/parse-config (detect/detect-schema db-spec) datasource)
                      {:dbwalk/data-sources {:db db-spec}})
        query-tree-with-sources (utils/with-datasource datasource query-tree)]
    (deref
      (crawler/run-query config query-tree-with-sources))))


(defn query-fn
  "Detects the database schema, returns a function for querying it."
  [db-spec]
  (let [datasource {:data-source :db
                    :type        :dbwalk/source-sql}
        relations (:dbwalk/relations (config/parse-config (detect/detect-schema db-spec) datasource))]
    (fn q
      ([db-spec-param query-tree]
       (->> query-tree
            (utils/with-datasource datasource)
            (crawler/run-query {:dbwalk/relations    relations
                                :dbwalk/data-sources {:db db-spec-param}})
            (deref)
            (out-map/to-map)))
      ([query-tree]
       (q db-spec query-tree)))))


(defn simple-query->graph
  "Takes a Configuration and a query without datasource information. Uses the first datasource found in the Configuration.
  Returns the Loom graph, which can then be given to an output formatter."
  [configuration query-without-datasource]
  (let [data-source-name (first (keys (:dbwalk/data-sources configuration)))
        data-source (utils/default-datasource data-source-name)]
    (s/with-fn-validation
      (->> (crawler/run-query configuration (utils/with-datasource data-source query-without-datasource))
           (deref)))))


(defn filtered-getter
  "Takes a Configuration and a query without datasource information. Uses the first datasource found in the Configuration.
  Uses filtered output formatter."

  [configuration columns query-without-datasource]
  (filtered/dump-graph-as-filtered-map columns
                                       (simple-query->graph configuration query-without-datasource)))

(defn- first-namespace [cols]
  (->> cols
       (first)
       (namespace)
       (keyword)))

(defn get-filtered-with-auto-path
  "Takes a Configuration, requested columns as a vector of namespaced keys (:table/column) and a map of {table name -> HoneySQL query}.
  Automatically generates the QueryPath starting from the table of the first key in the vector. Uses the filtered input and output functions."
  (
   [configuration columns base-queries]
   (let [query-path (subgraph/build-path-from configuration (first-namespace columns))
         datasource (utils/default-datasource (first (keys (:dbwalk/data-sources configuration))))]
     (when query-path
       (let [query (utils/with-datasource datasource (filtered/query-for-columns query-path columns base-queries))]
         (filtered-getter configuration columns query)))))
  (
   [configuration columns]
   (get-filtered-with-auto-path configuration columns {})))


(s/defn get-filtered
  (
    [configuration :- crawler/Configuration
     columns :- [s/Keyword]
     query-path :- tree-map/QueryPath]
    (filtered-getter configuration columns (filtered/query-for-columns query-path columns)))
  (
    [configuration columns query-path base-queries]
    (filtered-getter configuration columns (filtered/query-for-columns query-path columns base-queries))))


(defn- config-for [db-id db-spec]
  (let [database-description (detect/detect-schema db-spec)
        configuration (merge (config/parse-config database-description (utils/default-datasource db-id))
                             {:dbwalk/data-sources {db-id db-spec}})]
    configuration))

(defn db-spec->configuration "Creates a simple Configuration from a db-spec."
  [db-spec]
  (config-for (keyword (gensym "simple-db-")) db-spec))


;; These (get-my-stuff*) WILL be removed before release.
(defn get-my-stuff-as "Test function. Detects schema every time."
  [db-spec query-description output-xfrom]
  (let [database-description (detect/detect-schema db-spec)
        configuration (merge (config/parse-config database-description {:data-source :test-db
                                                                        :type        :dbwalk/source-sql})
                             {:dbwalk/data-sources {:test-db db-spec}})]
    (s/with-fn-validation
      (-> (crawler/run-query configuration query-description)
          (deref)
          (output-xfrom)))))

(defn get-my-stuff "Test function. Detects schema every time."
  [db-spec query-description]
  (get-my-stuff-as db-spec query-description out-map/to-map))

(defn get-my-stuff-flat "Test function. Detects schema every time."
  [db-spec query-description]
  (get-my-stuff-as db-spec query-description flat/dump))
