(ns vincit.dbwalk.output.filtered
  "Implements the 'filtered' output function, which allows selecting only required attributes and tables."
  (:require
    [vincit.dbwalk.graph :as graph]
    [vincit.dbwalk.api.tree-map :as tree]
    [schema.core :as s]
    [plumbing.core :as pl]
    [vincit.dbwalk.entity-node :as en]
    [vincit.dbwalk.relations :as relation]
    [loom.graph :as lg]))

(def ^:private FilterData {s/Keyword #{s/Keyword}})

(s/defn ^:private target-table
  [graph :- (s/protocol lg/Digraph)
   from :- en/EntityNode
   to :- en/EntityNode]
  (relation/target-table (graph/relation-for-link graph from to)))

(defn- split-namespaced-keywords
  "Splits :namespace/name to [:namespace :name]"
  [kws]
  (map (juxt (comp keyword namespace) (comp keyword name)) kws))

(defn- kw-pairs-to-set [kw-pairs]
  (->> kw-pairs
       (map second)
       (apply hash-set)))

(s/defn ^:private columns-to-filter-data :- FilterData
  [columns :- [s/Keyword]]
  (->> columns
       (split-namespaced-keywords)
       (group-by first)
       (pl/map-vals kw-pairs-to-set)))


(s/defn ^:private hierarchy-dump-graph-from-node
  "Recursively dump subgraph starting from node."
  [filter-data :- FilterData
   graph :- (s/protocol lg/Digraph)
   start-node :- en/EntityNode])

(s/defn merge-children
  [filter-data :- s/Any
   graph :- (s/protocol lg/Digraph)
   start-node :- en/EntityNode])


(s/defn keep-or-ignore-node
  "If this node is to be kept, returns {tablename -> node}.
   Otherwise, returns this node's children."
  [filter-data :- s/Any
   graph :- (s/protocol lg/Digraph)
   node :- en/EntityNode]
  (let [tables-to-keep (apply hash-set (keys filter-data))]
    (if (contains? tables-to-keep (graph/table-for-entity graph node))
      (hash-map (graph/table-for-entity graph node)
                [(hierarchy-dump-graph-from-node filter-data graph node)])
      (merge-children filter-data graph node))))

(s/defn merge-children
  "Sorts and filters this node's children."
  [filter-data :- s/Any
   graph :- (s/protocol lg/Digraph)
   start-node :- en/EntityNode]
  (if-let [children (seq (loom.graph/successors graph start-node))]
    (->> children
         (graph/to-insertion-order graph)
         (map #(keep-or-ignore-node filter-data graph %))
         (apply merge-with #(vec (concat %1 %2))))))

(defn- select-columns [e cols]
  (if (contains? cols :*)
    e
    (select-keys e cols)))

(s/defn ^:private hierarchy-dump-graph-from-node
  "Recursively dump subgraph starting from node."
  [filter-data :- FilterData
   graph :- (s/protocol lg/Digraph)
   start-node :- en/EntityNode]
  (let [this-node-table (graph/table-for-entity graph start-node)
        columns-for-this-table (get filter-data this-node-table)
        filtered-start-node (select-columns (en/data-only start-node) columns-for-this-table)]
    (->> (merge-children filter-data graph start-node)
         (en/data-only)
         (merge filtered-start-node))))


(s/defn dump-graph-as-filtered-map
  "Returns the graph as a list of root entities, where each entity
  is merged with { tablename -> seq of eagerly fetched items for each :eager query. }"
  [columns :- [s/Keyword]
   graph :- (s/protocol lg/Digraph)]
  (let [filter-data (columns-to-filter-data columns)]
    (->> graph
         (graph/find-root-items)
         (graph/to-insertion-order graph)
         (map #(hierarchy-dump-graph-from-node filter-data graph %))
         (vec))))

;; Input function, will be moved.

(s/defn query-for-columns
  "Generates a minimal QueryTree using the QueryPath so that
  - all columns are included in the result
  - the queries in the base-queries map {table -> HoneySQL query} are merged with the generated QueryTree nodes.
  SELECT and FROM clauses will be generated solely based on columns-to-select
  and any such clauses in partial-queries-by-table will be ignored.
  Returns nil when a QueryTree cannot be generated."
  (
    [path :- tree/QueryPath
     columns :- [s/Keyword]
     base-queries :- {s/Keyword s/Any}]
    (let [filter-data (columns-to-filter-data columns)
          nodes (apply hash-set (keys filter-data))]
      (tree/query-from-path (tree/covering-subtree path nodes) filter-data base-queries)))

  (
    [path :- tree/QueryPath
     columns :- [s/Keyword]]
    (query-for-columns path columns {})))


