(ns vincit.dbwalk.api.tree-map
  "Simple Tree implementation for generating QueryPaths."
  (:require [schema.core :as s]
            [honeysql.helpers :as sql]))


(def MultiMap {s/Keyword #{s/Keyword}})

(def QueryPath {:start    (s/maybe s/Keyword)               ;; Start node
                :link-map MultiMap})                        ;; Node to successors

(s/defn empty-tree :- QueryPath
  []
  {:start    nil
   :link-map {}})

(s/defn ^:private add :- MultiMap
  "Adds a key-value pair to the multimap."
  [mm :- MultiMap
   k :- s/Keyword
   v :- s/Keyword]
  (assoc mm k (conj (get mm k #{}) v)))

(s/defn ^:private del :- MultiMap
  "Removes a key-value pair from the multimap."
  [mm :- MultiMap
   k :- s/Keyword
   v :- s/Keyword]
  (let [new-value (disj (get mm k) v)]
    (if (empty? new-value)
      (dissoc mm k)
      (assoc mm k new-value))))

(defn- start-from-if-empty [m from]
  (if (nil? (:start m))
    (assoc m :start from)
    m))

(defn- clear-from-if-empty [m]
  (if (empty? (:link-map m))
    (assoc m :start nil)
    m))


(s/defn nodes :- #{s/Keyword}
  [tree :- QueryPath]
  (let [tree-map (:link-map tree)]
    (apply hash-set (flatten (concat (keys tree-map) (map seq (vals tree-map)))))))

(s/defn empty-tree? [tree :- QueryPath]
  (nil? (:start tree)))

(s/defn contains-node? [tree :- QueryPath
                        node :- s/Keyword]
  (contains? (nodes tree) node))

(s/defn allowed-link? "Checks that this link does not create a loop in the graph
                       and that either the tree is empty or the source of the new link
                       is already included in the tree."
  [tree :- QueryPath
   from :- s/Keyword
   to :- s/Keyword]
  (cond
    (empty-tree? tree) true
    (not (contains-node? tree from)) false
    (contains-node? tree to) false
    :else true))

(defn link-to-leaf? "Checks that the link being removed does not have successors."
  [tree to]
  (not (contains? (:link-map tree) to)))

(s/defn add-link :- QueryPath
  [tree :- QueryPath
   from :- s/Keyword
   to :- s/Keyword]
  (if (allowed-link? tree from to)
    (-> tree
        (start-from-if-empty from)
        (update :link-map add from to))
    tree))

(s/defn remove-link :- QueryPath
  [tree :- QueryPath
   from :- s/Keyword
   to :- s/Keyword]
  (if (link-to-leaf? tree to)
    (-> tree
        (update :link-map del from to)
        (clear-from-if-empty))
    tree))

(s/defn root
  [tree :- QueryPath]
  (:start tree))

(s/defn successors
  [tree :- QueryPath
   node :- s/Keyword]
  (get (:link-map tree) node #{}))

(defn- invert-single-mapping [[k vs]]
  (for [v vs]
    [v k]))

(s/defn ^:private invert-path :- {s/Keyword s/Keyword}
  "Inverts all links in the QueryPath to create a map of successor -> predecessor."
  [tree :- QueryPath]
  (->> tree
       :link-map
       (mapcat invert-single-mapping)
       (into {})))

(s/defn ^:private bfs :- (s/maybe s/Keyword)
  "Finds the node (in nodes) closest to the root of the QueryPath.
   Returns nil if more than one node is at the same distance or if none of the nodes are in the QueryPath."
  [path :- QueryPath
   nodes :- #{s/Keyword}]
  (when (not (empty-tree? path))
    (loop [succs #{(root path)}]
      (when (seq succs)
        (let [found-nodes (clojure.set/intersection succs nodes)
              found-count (count found-nodes)]
          (cond
            (= 1 found-count) (first found-nodes)
            (= 0 found-count) (recur (apply clojure.set/union (map #(successors path %) succs)))
            :else nil))))))                                 ;; More than one found at the same time -> neither can be root

(s/defn ^:private walk-upto :- [s/Keyword]
  "Find the path from node to target, which should be among the predecessors of node."
  [inverted-path :- {s/Keyword s/Keyword}
   target :- s/Keyword
   node :- s/Keyword]
  (loop [nodes (vector node)]
    (when-let [next-node (get inverted-path (last nodes))]
      (let [path-to-next (conj nodes next-node)]
        (if (= next-node target)
          path-to-next
          (recur path-to-next))))))

(defn- add-path [query-path path-to-add]
  (if (> (count path-to-add) 1)
    (reduce #(apply add-link %1 %2) query-path (partition 2 1 path-to-add))
    query-path))

(s/defn covering-subtree :- (s/maybe QueryPath)
  "Finds the smallest subtree which contains the given nodes,
   one of which must be the starting node of the resulting subtree.
   Returns nil if no such subtree can be found."
  [path :- QueryPath
   nodes :- #{s/Keyword}]
  (let [inverted-path (invert-path path)
        start-node (bfs path nodes)]
    (when start-node
      (let [leaf-nodes (disj nodes start-node)
            paths-to-leaf-nodes (->> leaf-nodes
                                     (map (partial walk-upto inverted-path start-node))
                                     (map reverse))]
        (when (not-any? empty? paths-to-leaf-nodes)
          (reduce add-path (empty-tree) paths-to-leaf-nodes))))))


(defn- add-select [query columns-as-seq]
  (apply sql/select query columns-as-seq))


(defn- subtree [m node columns-to-select base-queries]
  (let [successors (get-in m [:link-map node])
        columns (get columns-to-select node #{:*})
        main-query {:query (-> (get base-queries node {})
                               (add-select  columns)
                               (sql/from node))}]
    (if (seq successors)
      (merge main-query {:eager (mapv #(subtree m % columns-to-select base-queries) successors)})
      main-query)))

(s/defn query-from-path
  "Generates a minimal QueryTree using the QueryPath so that
    - all columns in columns-to-select are included in the result
    - the partial queries {table->HoneySQL query} are merged with the generated QueryTree nodes.
    SELECT and FROM clauses will be generated solely based on columns-to-select
    and any such clauses in partial-queries-by-table will be ignored."
  (
    [tree :- QueryPath
     columns-to-select :- {s/Keyword #{s/Keyword}}
     partial-queries-by-table :- {s/Keyword s/Any}]
    (subtree tree (:start tree) columns-to-select partial-queries-by-table))
  (
    [tree :- QueryPath
     columns-to-select :- {s/Keyword #{s/Keyword}}]
    (subtree tree (:start tree) columns-to-select {}))
  (
    [tree :- QueryPath]
    (subtree tree (:start tree) nil {})))


