(ns vincit.dbwalk.database.sql
  "An implementation for the protocols and multimethods required to use an SQL database as a datasource."
  (:require [clojure.java.jdbc :as jdbc]
            [honeysql.helpers :as hsql]
            [honeysql.core :as sql]
            [vincit.dbwalk.relations :as relations]
            [vincit.dbwalk.query-parser :as parser]))


(defrecord ColumnSpec [table column]
  relations/RelationalData
  (property-name [this]
    column)
  (table [this]
    table))

(defn- create-where-clause [this source-items]
  (hsql/where [:in (relations/target-property this) (map (relations/source-property this) source-items)]))

(defn- add-merge-clauses-for-relations "Generates the HoneySQL query for SELECTing the next level in the query result."
  [query-node joins-to-follow nodes-to-advance]
  (let [relation-clauses (->> (map #(create-where-clause % nodes-to-advance) joins-to-follow)
                              (mapv :where)
                              (into [:or]))
        query (parser/query query-node)]
    ;; Just cannot figure out HoneySQL's merge-where
    (if (:where query)
     (hsql/merge-where (dissoc query :where) [:and (:where query) relation-clauses])
     (hsql/merge-where query relation-clauses))))

(defn- add-select-clause-for-properties [query required-properties]
  (apply hsql/select query (seq required-properties)))

(defmethod relations/advance-query-tree :dbwalk/source-sql [config query-node relations-to-follow nodes-to-advance properties-to-get]
  (let [data-source (relations/data-source-from-relations relations-to-follow)
        data-source-config (relations/select-data-source-from-configuration config data-source)
        query (-> query-node
                  (add-merge-clauses-for-relations relations-to-follow nodes-to-advance)
                  (add-select-clause-for-properties properties-to-get)
                  (#(relations/pre-query-filter config data-source % nodes-to-advance relations-to-follow))
                  (sql/format))]
    (jdbc/query data-source-config query)))

(defmethod relations/read-from-table :dbwalk/source-sql [configuration data-source query properties-to-select relations-to-follow]
  (let [query (apply hsql/select query (seq properties-to-select))
        data-source-config (relations/select-data-source-from-configuration configuration data-source)
        filtered-query (relations/pre-query-filter configuration data-source query [] relations-to-follow)]
    (jdbc/query data-source-config (sql/format filtered-query))))



(defmethod relations/insert-to-table :dbwalk/source-sql [configuration data-source-description table entity]
  (let [db-spec (get-in configuration [:dbwalk/data-sources (:data-source data-source-description)])
        primary-keys (get-in configuration [:dbwalk/primary-keys data-source-description])
        primary-key-column (get primary-keys table)
        insert-result (jdbc/insert! db-spec table entity)
        primary-key (get (first insert-result) primary-key-column)]
    primary-key))
