(ns vincit.dbwalk.api.sweet
  (:require [vincit.dbwalk.graph :as graph]
            [vincit.dbwalk.utils :as utils]
            [vincit.dbwalk.schema-detect :as detect]
            [vincit.dbwalk.config :as config]
            [vincit.dbwalk.crawler :as crawler]
            [vincit.dbwalk.output.flat :as flat]
            [vincit.dbwalk.output.filtered :as filtered]
            [vincit.dbwalk.output.map :as out-map]
            [schema.core :as s]))

(defn query-single-db
  "Runs the given query. Returns a Loom graph."
  [db-spec query-tree]
  (let [datasource {:data-source :db
                    :type        :dbwalk/source-sql}
        config {:dbwalk/relations    (config/parse-config (detect/detect-schema db-spec) datasource)
                :dbwalk/data-sources {:db db-spec}}
        query-tree-with-sources (utils/with-datasource datasource query-tree)]
    (deref
      (crawler/run-query config query-tree-with-sources))))


(defn query-fn [db-spec]
  (let [datasource {:data-source :db
                    :type        :dbwalk/source-sql}
        relations (config/parse-config (detect/detect-schema db-spec) datasource)]
    (fn q
      ([db-spec-param query-tree]
       (->> query-tree
            (utils/with-datasource datasource)
            (crawler/run-query {:dbwalk/relations    relations
                                :dbwalk/data-sources {:db db-spec-param}})
            (deref)
            (out-map/to-map)))
      ([query-tree]
       (q db-spec query-tree)))))


(defn config-for [db-id db-spec]
  (let [database-description (detect/detect-schema db-spec)
        configuration {:dbwalk/relations    (config/parse-config database-description {:data-source db-id
                                                                                       :type        :dbwalk/source-sql})
                       :dbwalk/data-sources {db-id db-spec}}]
    configuration))


(defn get-filtered [configuration columns query-path]
  (let [data-source-name (first (keys (:dbwalk/data-sources configuration)))
        data-source (hash-map :data-source data-source-name
                              :type :dbwalk/source-sql)
        query (filtered/query-for-columns query-path columns)]
    (s/with-fn-validation
      (->> (crawler/run-query configuration (utils/with-datasource data-source query))
           (deref)
           (filtered/dump-graph-as-filtered-map columns)))))

(defn get-my-stuff-as "Test function. Detects schema every time."
  [db-spec query-description output-xfrom]
  (let [database-description (detect/detect-schema db-spec)
        configuration {:dbwalk/relations    (config/parse-config database-description {:data-source :test-db
                                                                                       :type        :dbwalk/source-sql})
                       :dbwalk/data-sources {:test-db db-spec}}]
    (s/with-fn-validation
      (-> (crawler/run-query configuration query-description)
          (deref)
          (output-xfrom)))))

(defn get-my-stuff "Test function. Detects schema every time."
  [db-spec query-description]
  (get-my-stuff-as db-spec query-description out-map/to-map))

(defn get-my-stuff-flat "Test function. Detects schema every time."
  [db-spec query-description]
  (get-my-stuff-as db-spec query-description flat/dump))
