(ns vincit.dbwalk.relations
  "Protocols and default implementations for relations.")

(defprotocol RelationalData
  (property-name [this])
  (table [this]))

(defprotocol Relation
  (get-inverse-join [this])
  ;; Functions for reader
  (created-attribute-name [this] "Name of the key in the result map for this join's related items.")
  (required-attribute-for-join [this])
  ;; Functions for insert
  (foreign-key-name [this] "The column name for the foreign key used in this relation.")
  (insert-source-first [this] "Returns true if the foreign key is in the target table of this relation."))


;; Forward declaration
(def ->OneToMany)

(defn from-data-source [this]
  (:from-datasource this))

(defn to-data-source [this]
  (:to-datasource this))

(defn from-description [this]
  (:from-description this))

(defn to-description [this]
  (:to-description this))

(defn source-table [this]
  (table (from-description this)))
(defn source-property [this]
  (property-name (from-description this)))
(defn target-table [this]
  (table (to-description this)))
(defn target-property [this]
  (property-name (to-description this)))


(defrecord ManyToOne [from-datasource from-description to-datasource to-description]
  Relation
  (get-inverse-join [this]
    (->OneToMany to-datasource to-description from-datasource from-description))
  (created-attribute-name [this]
    (source-property this))
  (required-attribute-for-join [this]
    (property-name from-description))
  (insert-source-first [this]
    false)
  (foreign-key-name [this]
    (source-property this)))

(defrecord OneToMany [from-datasource from-description to-datasource to-description]
  Relation
  (get-inverse-join [this]
    (->ManyToOne to-datasource to-description from-datasource from-description))
  (created-attribute-name [this]
    (target-table this))
  (required-attribute-for-join [this]
    (property-name from-description))
  (insert-source-first [this]
    true)
  (foreign-key-name [this]
    (target-property this)))

(defn data-source-from-relations [relations]
  (-> relations
      first
      to-data-source))

(defn select-data-source-from-configuration [config data-source]
  (let [ds-name (:data-source data-source)
        data-sources (:dbwalk/data-sources config)]
    (get data-sources ds-name)))

(defmulti advance-query-tree "Reads the next level of data for the query tree."
          (fn [config query-node relations-to-follow nodes-to-advance properties-to-get]
            (-> relations-to-follow
                data-source-from-relations
                :type)))

;; "Table" is used to describe not only database tables but also other kinds of namespaces.

(defmulti read-from-table "Executes the query in the datasource, additionally selecting the given properties."
          (fn [configuration datasource query properties-to-select relations-to-follow]
            (:type datasource)))

(defmulti insert-to-table "Inserts the entity to the table in the data-source."
          (fn [configuration data-source-description table entity]
            (:type data-source-description)))


(defmulti pre-query-filter
          "Hook function for modifying queries before datasource access. Can be used for authorization etc.
           Must return a QueryTree node."
          (fn [config datasource query current-entities relations]
            datasource))

(defmethod pre-query-filter :default
  [config datasource query current-entities relations]
  query)
