(ns vincit.dbwalk.api.subgraph
  "REPL utility for creating QueryPaths.
   A QueryPath is a directed path which can contain each table at most once."
  (:require [vincit.dbwalk.schema-detect :as detect]
            [vincit.dbwalk.config :as config]
            [vincit.dbwalk.api.tree-map :as tree]
            [vincit.dbwalk.relations :as relations]
            [plumbing.core :as pl]
            [schema.core :as s]
            [vincit.dbwalk.query-parser :refer [QueryTree]]))

(def Builder {:data-source {:data-source s/Keyword
                            :type        s/Keyword}
              :config      {:dbwalk/relations s/Any
                            :dbwalk/data-sources     s/Any}
              :subgraph    tree/QueryPath})

(s/defn builder-for :- (s/atom Builder)
  "Detects db schema and returns a Builder atom."
  [db-spec]
  (let [datasource {:data-source :db
                    :type        :dbwalk/source-sql}
        config {:dbwalk/relations (config/parse-config (detect/detect-schema db-spec) datasource)
                :dbwalk/data-sources     {:db db-spec}}
        empty-subgraph (tree/empty-tree)]
    (atom {:data-source datasource
           :config      config
           :subgraph    empty-subgraph})))

(defn- all-relations [builder]
  (get-in builder [:config :dbwalk/relations (:data-source builder)]))

(defn- all-tables [builder]
  (->> (all-relations builder)
       (map relations/source-table)
       (apply hash-set)))

(defn- all-relations-as-sets [builder]
  (->> (all-relations builder)
       (map #(hash-set (relations/source-table %) (relations/target-table %)))
       (apply hash-set)))


(s/defn sources :- #{s/Keyword}
  "Lists tables the QueryPath can be extended from."
  [builder-atom :- (s/atom Builder)]
  (let [builder (deref builder-atom)]
    (if (tree/empty-tree? (:subgraph builder))
      (all-tables builder)
      (tree/nodes (:subgraph builder)))))

(s/defn targets :- #{s/Keyword}
  "Lists tables the QueryPath can extend to starting from the given table."
  [builder-atom :- (s/atom Builder)
   from :- s/Keyword]
  (let [builder (deref builder-atom)]
    (->> (all-relations builder)
         (filter #(= from (relations/source-table %)))
         (filter #(tree/allowed-link? (:subgraph builder) from (relations/target-table %)))
         (map relations/target-table)
         (apply hash-set))))

(defn- set-of-targets [xs]
  (apply hash-set (map relations/target-table xs)))

(s/defn possible-links :- {s/Keyword #{s/Keyword}}
  "Lists all possible new links for extending the QueryPath."
  [builder-atom :- (s/atom Builder)]
  (let [builder (deref builder-atom)]
    (->> (all-relations builder)
         (filter #(tree/allowed-link? (:subgraph builder) (relations/source-table %) (relations/target-table %)))
         (group-by relations/source-table)
         (pl/map-vals set-of-targets))))


(s/defn add-link
  [builder-atom :- (s/atom Builder)
   from :- s/Keyword
   to :- s/Keyword]
  (let [builder (deref builder-atom)
        possibles (all-relations-as-sets builder)]
    (if (contains? possibles #{from to})
      (swap! builder-atom update :subgraph #(tree/add-link % from to)))))

(s/defn remove-link
  [builder-atom :- (s/atom Builder)
   from :- s/Keyword
   to :- s/Keyword]
  (swap! builder-atom update :subgraph #(tree/remove-link % from to)))

(s/defn query-path :- tree/QueryPath
  [builder-atom :- (s/atom Builder)]
  (-> builder-atom
      (deref)
      (:subgraph)))

