(ns vincit.dbwalk.crawler
  "Implements 'walking' through the datasources."
  (:require [vincit.dbwalk.query-parser :as parser]
            [vincit.dbwalk.relations :as relations]
            [vincit.dbwalk.graph :as graph]
            [vincit.dbwalk.database.sql]
            [vincit.dbwalk.entity-node :as en]
            [schema.core :as s]
            [clojure.tools.logging :as log]))


(def Configuration {:dbwalk/relations    {(s/pred map?) [(s/protocol relations/Relation)]}
                    :dbwalk/data-sources s/Any
                    s/Keyword            s/Any})            ;; Allow extension

(defn- merge-selected-properties [all-columns]
  (disj
    (if (or (empty? all-columns)
            (contains? all-columns :*))
      #{:*}
      all-columns)
    nil))

(defn- add-required-properties "Combines properties required for joins with properties selected by the user
                                to a list that will be used in the SELECT clause.
                                If no columns are selected, selects all columns."
  [& props]
  (let [all-properties (apply hash-set (flatten (concat props)))]
    (merge-selected-properties all-properties)))


(defn kv-filter [k v]
  (fn [x]
    (= v (get x k))))

(defn select-related-items [relation source possible-targets]
  (let [key (relations/target-property relation)
        value ((relations/source-property relation) source)]
    (filter (kv-filter key value) possible-targets)))

(s/defn ^:private distribute-children :- (s/maybe {en/EntityNode [en/EntityNode]})
  "Relate the new items to their parents.
   Essentially the same as a JOIN between the parent and children tables, but with maps."
  [parents :- [en/EntityNode]
   children :- [(s/pred map?)]
   relations :- [(s/protocol relations/Relation)]]
  (apply merge
         (for [e parents]
           (apply merge-with concat
                  (for [rel relations]
                    (let [bare-entity (:data e)]
                      {e
                       (en/wrap-entities rel (select-related-items rel bare-entity children))}))))))

(s/defn ^:private find-joins-for-query-step :- [(s/protocol relations/Relation)]
  "Finds the Relations between two sources of data.
   There can be more than one if source table is linked to the target table from two or more columns."
  [config :- s/Any
   query-tree :- parser/QueryTree
   next-node :- parser/QueryTree]
  (let [this-data-source (parser/data-source-and-type query-tree)
        {relations-by-source :dbwalk/relations} config
        possible-relations (get relations-by-source this-data-source [])]
    (->> possible-relations
         (filter #(parser/compare-target-with-query next-node %))
         (filter #(parser/compare-source-with-query query-tree %)))))

(s/defn ^:private columns-required-for-forward-joins :- [s/Keyword]
  "Lists columns which have to be included in this query level (in next-level items!) so that next level join is possible."
  [configuration :- Configuration
   query-node :- parser/QueryTree]
  (flatten
    (concat
      (for [target (parser/children query-node)]
        (->> (find-joins-for-query-step configuration query-node target)
             (map relations/source-property))))))

(defn- get-items [config nodes-to-advance query-node joins-to-follow]
  (if (not (seq joins-to-follow))
    (log/error "Missing target from a relation in " query-node)
    (let [required-forward-columns (columns-required-for-forward-joins config query-node)
          required-join-columns (map relations/target-property joins-to-follow)
          columns-from-query-tree (parser/selected query-node)
          properties-to-select (add-required-properties required-forward-columns required-join-columns columns-from-query-tree)]
      (relations/advance-query-tree config query-node joins-to-follow nodes-to-advance properties-to-select))))


(s/defn ^:private advance-query-tree "Walks the query tree and builds the query result graph."
  [config :- Configuration
   query-tree :- parser/QueryTree
   graph :- (s/atom (s/protocol loom.graph/Digraph))
   nodes-to-advance :- [(s/pred map?)]]
  (when (seq nodes-to-advance)
    (doall
      (for [next-query-node (parser/children query-tree)]
        (let [joins-to-follow (find-joins-for-query-step config query-tree next-query-node)
              bare-entities (en/unwrap-entities nodes-to-advance)
              new-items (get-items config bare-entities next-query-node joins-to-follow)
              entity-to-new-children (distribute-children nodes-to-advance new-items joins-to-follow)
              new-entity-nodes (apply concat (vals entity-to-new-children))]
          (graph/add-children! graph entity-to-new-children)
          (advance-query-tree config next-query-node graph new-entity-nodes))))))

(defn fake-relation "Creates a relation to use as the (fake) source relation for first level items."
  [data-source query-tree]
  (relations/->ManyToOne nil nil data-source (reify relations/RelationalData
                                               (property-name [_])
                                               (table [_]
                                                 (parser/from query-tree)))))

(s/defn run-query "Reads the requested data from the db and returns it as a graph.
                 The result graph will have one node per db row.
                 The links between nodes have the table->table relation as an attribute."
  [configuration :- Configuration
   query-tree :- parser/QueryTree]
  (let [storage (graph/new-graph)
        initial-query (parser/query query-tree)
        required-columns (columns-required-for-forward-joins configuration query-tree)
        properties-to-select (add-required-properties required-columns (parser/selected query-tree))
        initial-data-source (parser/data-source-and-type query-tree)
        fake-first-relation (fake-relation initial-data-source query-tree)
        first-level-items (relations/read-from-table configuration initial-data-source initial-query properties-to-select (list fake-first-relation))
        wrapped-first-level-items (en/wrap-entities fake-first-relation first-level-items)]

    (graph/add-first-level-entities! storage wrapped-first-level-items)
    (advance-query-tree configuration query-tree storage wrapped-first-level-items)
    storage))
