(ns webaudio.noise
  (:require [webaudio.audio-api :as api]))

(defn create-mult-wave-buffer 
  "Generate a multi-channel wave buffer. The wave signal is represetned by the function f
  http://chimera.labs.oreilly.com/books/1234000001552/ch06.html#s06_3"
  [node f]
  (let [buffer (api/get-buffer node)
        sample-rate (api/get-sample-rate buffer)
        lengthInSamples (api/get-length-in-samples buffer)
        num-channels (api/get-number-of-channels buffer)]
    (dotimes [j num-channels]
      (let [data (api/get-channel-data buffer j)]
        (dotimes [i lengthInSamples]
          (aset data i (f i sample-rate)))))))

(defn create-mult-wave-node
  [context f &
    {:keys [length-in-seconds num-channels start?] 
       :or {length-in-seconds 5 num-channels 1 start? false}}]
  (let [sample-rate (api/sample-rate context)
        lengthInSamples (* length-in-seconds sample-rate)
        node (api/create-buffer-source context)
        buffer (api/create-buffer context num-channels lengthInSamples sample-rate)]
    (api/set-buffer node buffer)
    (create-mult-wave-buffer node f)
    (api/set-loop node true)
    (when start?
      (api/start node))
    node))

(defn create-white-noise-node 
  [context &{:keys [start? num-channels length-in-seconds] :or {start? false num-channels 1 length-in-seconds 5}}] 
  (create-mult-wave-node context #(- (rand 2) 1) :num-channels num-channels :start? start? :length-in-seconds length-in-seconds))

(defn fm-noise "http://en.wikipedia.org/wiki/Frequency_modulation" [t f fm fd-percentage]
  (Math/sin (+ (* 2 Math/PI f t) (* f (/ (/ fd-percentage 100) fm) (Math/sin (* 2 Math/PI t fm))))))

(defn update-fm-wave-node [node frequency x y]
  (create-mult-wave-buffer node #(fm-noise (/ %1 %2) frequency x y)))

(defn create-fm-wave-node
  [context frequency x y &{:keys [start? num-channels length-in-seconds] :or {start? false num-channels 1 length-in-seconds 1}}] 
  (create-mult-wave-node context #(fm-noise (/ %1 %2) frequency x y) :num-channels num-channels :start? start? :length-in-seconds length-in-seconds))
