(ns ventas-devtools.client
  "Client defstates"
  (:require
   [clojure.java.shell]
   [ventas-devtools.client.auto :as auto]
   [ventas-devtools.client.sassc :as sassc]
   [ventas-devtools.config :as devtools.config]
   [figwheel-sidecar.components.css-watcher :as figwheel.css-watcher]
   [figwheel-sidecar.repl-api :as figwheel]
   [figwheel-sidecar.config :as figwheel.config]
   [mount.core :refer [defstate]]
   [taoensso.timbre :as timbre]
   [ventas.config :as config]
   [ventas.plugin :as plugin]
   [ventas.theme :as theme]))

(def figwheel-theme (atom nil))

(alter-var-root
 #'figwheel.css-watcher/handle-css-notification
 (fn [_]
   (fn ventas-css-handler [watcher files]
     (when-let [changed-css-files (not-empty (filter #(.endsWith % ".css") (map str files)))]
       (let [figwheel-server (:figwheel-server watcher)
             sendable-files (map #(figwheel.css-watcher/make-css-file %) changed-css-files)]
         (figwheel.css-watcher/send-css-files figwheel-server sendable-files))))))

(defn- find-source-paths []
  (->> ["src/cljs" "src/cljc" "test/cljs" "test/cljc" "custom-lib"]
       (map clojure.java.io/file)
       (filter #(.exists %))
       (map str)))

(defn- dev-build []
  (let [theme-id (or @figwheel-theme (theme/current))
        theme (plugin/find theme-id)]
    {:id "app"
     :source-paths (find-source-paths)
     :figwheel {:on-jsload "ventas.core/on-figwheel-reload"}
     :compiler (merge {:asset-path (str "files/js/compiled/" (name theme-id))
                       :output-to (str "resources/public/files/js/compiled/" (name theme-id) ".js")
                       :output-dir (str "resources/public/files/js/compiled/" (name theme-id))
                       :source-map-timestamp true
                       :devcards true
                       :preloads ['devtools.preload]
                       :parallel-build true}
                      (:build theme))}))

(defn- get-project []
  (-> (figwheel.config/->lein-project-config-source)
      (figwheel.config/->config-data)
      :data
      (assoc :root (.getAbsolutePath (clojure.java.io/file ".")))))

(def ^:private default-figwheel-config
  {:css-dirs ["resources/public/files/css"]
   :server-logfile "log/figwheel.log"
   :repl false})

(defn figwheel-start []
  (when (config/get :embed-figwheel?)
    (let [build (dev-build)]
      (timbre/info "Starting Figwheel, main ns:" (get-in build [:compiler :main]))
      (figwheel/start-figwheel!
       (merge default-figwheel-config
              (:figwheel (get-project))
              {:builds [build]
               :builds-to-start ["app"]})))))

(defn figwheel-stop []
  (when (config/get :embed-figwheel?)
    (timbre/info "Stopping Figwheel")
    (figwheel/stop-figwheel!)))

(defstate figwheel :start (figwheel-start) :stop (figwheel-stop))

(defn compile-sass! []
  (let [theme-ids (get-in (get-project) [:ventas :themes])]
    (sassc/compile (devtools.config/sassc theme-ids))))

(defstate sass
  :start
  (do
    (timbre/info "Starting SASS")
    (auto/auto {:project (get-project)
                :config {:file-pattern #"\.(scss)$"
                         :paths ["src/scss"]}
                :identifier "sassc"
                :task-fn compile-sass!}))
  :stop
  (do
    (timbre/info "Stopping SASS")
    (future-cancel sass)))
