(ns vadelabs.fds.demo.events
  (:require
   [re-frame.core :refer [reg-event-db reg-event-fx reg-fx inject-cofx trim-v after path debug]]
   [day8.re-frame.tracing :refer-macros [fn-traced]]
   [vadelabs.fds.demo.db :as db]))

(def set-user-interceptor 
  "Every event handler can be 'wrapped' in a chain of interceptors. Each of these
   interceptors can do things 'before' and/or 'after' the event handler is executed.
   They are like the 'middlerware' of web servers, wrapping around the 'handler'.
   Interceptors are a useful way of factoring out commonality (across event handlers)
   and looking after cross-cutting concerns like logging or validation.

   They are also used to 'inject' values into the 'coeffects' parameter of an 
   an event handler, when that handlers needs access to certain resources.
   
   Each event handler can have its own chain of interceptors. Below we create the
   interceptor chain shared by all event handlers which manipulate user.
  
   Below we create the interceptor chain shared by all event handlers which manipulate user.
   A chain of interceptors is a vector."
  [(path :user)        ;; `:user` path within `db`, rather than the full `db`.
   (after db/set-user-ls) ;; write user to localstore (after)
   trim-v            ;; removes first (event id) element from the event vector 
   ])

(def remove-user-interceptor
  "After logging out clean up local-storage so that when a users refreshes
   the browser she/he is not automatically logged-in, and because it's a
   good practice to clean-up after yourself."
  [(after db/remove-user-ls)])

;; -- Helpers -------

;; -- Event Handlers --------

(reg-event-fx   ;; usage: (dispatch [:initialise-db])
 :initialize-db ;; sets up initial application state

 ;; the interceptor chain (a vector of interceptors)
 [(inject-cofx :local-store-user)] ;; gets user from localstore, and puts into coeffects arg

 ;; the event handler (function) being registered
 (fn-traced [{:keys [local-store-user]} _]                    ;; take 2 vals from coeffects. Ignore event vector itself.
            {:db (assoc db/default-db :user local-store-user)})) ;; what it returns becomes the new application state

(reg-event-fx     ;; usage: (dispatch [:initialise-db])
 :set-active-page ;; sets up initial application state

 ;; the interceptor chain (a vector of interceptors)
 [(inject-cofx :local-store-user)] ;; gets user from localstore, and puts into coeffects arg

 ;; the event handler (function) being registered
 (fn-traced [{:keys [local-store-user]} _]                    ;; take 2 vals from coeffects. Ignore event vector itself.
            {:db (assoc db/default-db :user local-store-user)})) ;; what it returns becomes the new application state
