(ns uswitch.ring-riemann
  (:require [riemann.client]
            [clojure.string :as string]
            [clj-stacktrace.repl :as stacktrace]))

;; utils

(defn mapply
  [f & args]
  (apply f (apply concat (butlast args) (last args))))

(defmacro suppress-exceptions
  [& body]
  `(try ~@body (catch Throwable t# nil)))

;; middleware

(defprotocol EventConstructor
  (success-event [_ req res time]
    "Given a ring request, a ring response and the time (in ms) taken,
    return riemann event data")
  (exception-event [_ req throwable time]
    "Given a ring request, an exception and the time (in ms) taken,
    return riemann event data"))

(defn default-event-constructor
  [service]
  (reify Object
    EventConstructor
    (success-event [_ req res time]
      {:service        service
       :state          (str (:status res))
       :metric         time})
    (exception-event [_ req throwable time]
      {:service        service
       :state          "error"
       :tags           ["exception"]
       :metric         time
       :description    (stacktrace/pst-str throwable)
       :exception-type (str (type throwable))})))


(defn- make-client
  [client-opts]
  (let [client-fn (if (= :udp (:protocol client-opts))
                    riemann.client/udp-client
                    riemann.client/tcp-client)]
    (mapply client-fn client-opts)))



(defn wrap-riemann
  [handler client-opts event-constructor]
  (let [client (delay (make-client client-opts))
        ack?   (get client-opts :ack false)]
    (fn [req]
      (let [t (System/currentTimeMillis)]
        (try
          (let [res (handler req)]
            (suppress-exceptions
             (riemann.client/send-event @client
                                        (success-event event-constructor req res (- (System/currentTimeMillis) t))
                                        ack?))
            res)
          (catch Throwable ex
            (suppress-exceptions
             (riemann.client/send-event @client
                                        (exception-event event-constructor req (- (System/currentTimeMillis) t) ex)
                                        ack?))
            (throw ex)))))))


(defn wrap-riemann-service
  [handler client-opts service]
  (wrap-riemann handler
                client-opts
                (default-event-constructor service)))
