# ring-riemann

**ring-riemann** will time the execution of a ring handler and send an
event to the specified riemann instance each time a request is
processed.

By default, succesfully processed requests will send events with the
following data:

- `service`: a user specified service name
- `state`: the http status code
- `metric`: the time in ms taken to complete the request

If an exception occurs in the wrapped handler, the following data will
instead be sent:

- `service`: a user specified service name
- `state`: "error"
- `tags`: ["exception"]
- `metric`: the time in ms taken until the exception occured
- `description`: a pretty-printed stack trace
- `exception-type`: the type of the thrown exception

This information can easily be customized.

## Version

![Version](https://clojars.org/uswitch/ring-riemann/latest-version.svg)

## Usage

When constructing the riemann ring handler, two arguments are always
required:

- `handler`: the ring handler fn to instrument
- `client-opts`: a map of options that will be used to construct the
  Clojure riemann client. Besides any options understood by the client
  function, the following keys can be specified:
  - `:protocol`: can be used to control the protocol used. Can be
    either `:tcp` or `:udp`, default is `:tcp`.
  - `:ack`: `true` or `false`, require request to be acknowledged or not.

The quickest way to get started is to use the `wrap-riemann-service`
middleware function. Besides an handler and riemann client options, it
requires a service name which will be used for the events dispatched.

```clojure
(require '[uswitch.ring-riemann :refer [wrap-riemann-service]])

(def my-instrumented-handler
  (wrap-riemann-service my-existing-handler
                        {:host "127.0.0.1" :port 5555}
                        "my-webscale-api"))
```

If you need full control over the events sent to riemann, you can use
the `wrap-riemann` middleware function instead, and provide your own
implementation of the `EventConstructor` protocol.

```clojure
(require '[uswitch.ring-riemann :refer [wrap-riemann]])

(def my-instrumented-handler
  (wrap-riemann my-existing-handler
                {:host "127.0.0.1"}
                (reify Object
                  EventConstructor
                  (success-event [_ req res time]
                    {:service        "my-webscale-api"
                     :state          (str (:status res))
                     :metric         time})
                  (exception-event [_ req throwable time]
                    {:service        "my-webscale-api"
                     :state          "error"
                     :tags           ["exception"]
                     :metric         time
                     :description    (stacktrace/pst-str throwable)
                     :exception-type (str (type throwable))}))))
```





## License

Copyright Ragnar Dahlen © 2013

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.
