(ns uswitch.boot-containers.ecs
  (:require
   [amazonica.aws.ecs :as ecs]
   [boot.core :as boot]))

(def ^{:doc "
env expects an environment with the shape:

{:aws {:creds  {:aws-access-key \"...\"
                :aws-secret-key \"...\"
                :endpoint \"eu-west-1\"}
       :ecs    {:cluster \"cluster-name\" :family \"task-name\"}
       :docker {:registry \"...\"
                :api-version \"v1\"}}"}
  env
  (let [{:keys [creds] {:keys [cluster family]} :ecs} (boot/get-env :aws)]
    {:creds creds :cluster cluster :family family}))

(defn tasks
  "Returns a list of ECS tasks on cluster `cluster` with name `family`."
  [{:keys [creds cluster family]}]
  (:task-arns (ecs/list-tasks creds :cluster cluster :family family)))

(defn stop-task!
  "Stops task with id `task` on cluster `cluster`."
  [{:keys [creds cluster]} task]
  (ecs/stop-task creds :cluster cluster :task task))

(defn stop-tasks!
  "Stops all tasks on cluster `cluster`, named `family`"
  [{:keys [creds cluster family] :as opts}]
  (doseq [task (tasks opts)]
    (stop-task! opts task)))

(defn services
  "Returns a list of services on cluster `cluster`."
  [{:keys [creds cluster]}]
  (:service-arns (ecs/list-services creds :cluster cluster)))

(defn describe-services
  "Returns a list of services descriptions on cluster `cluster`."
  [{:keys [creds cluster]}]
  (ecs/describe-services creds
                         :cluster cluster
                         :services (services cluster)))

(defn deployments
  "Returns a list of deployments on cluster `cluster` named `family`"
  [{:keys [creds cluster family]}]
  (->> (describe-services cluster)
       :services
       (filter (comp #{family} :service-name))
       first
       :deployments))

(defn image-id
  [registry family tag]
  (format "%s/%s:%s" registry family tag))

(defn container-definition
  [name & {:keys [image] :as opts}]
  (merge {:volumesFrom []
          :mountPoints []
          :name name
          :entryPoint []
          :memory 500
          :command []
          :portMappings []
          :image image
          :environment []
          :essential true
          :cpu 100
          :links []}
         opts))

(defn register-task-definition!
  [{:keys [creds cluster family]}]
  (ecs/register-task-definition creds
                                {:family family
                                 :container-definitions [container-definition]}))

(defn update-service!
  [{:keys [creds cluster family]}]
  (let [task-rev (-> creds
                     (ecs/describe-task-definition :task-definition family)
                     :task-definition
                     :revision)]
    (ecs/update-service creds
                        :cluster cluster
                        :service family
                        :task-definition (format "%s:%s" family task-rev)
                        :desired-count 1)))
