(ns uswitch.boot-containers
  (:require
   [boot.core :as boot]
   [boot.pod :as pod]
   [boot.task.built-in :as task]))

(def dependencies
  [['boot/core (boot.App/getBootVersion)]
   ['amazonica "0.3.24"]
   ['clj-http "1.1.2"]
   ['org.clojure/data.json "0.2.6"]])

(def pod
  (-> (boot/get-env)
      (update-in [:dependencies] concat dependencies)
      (pod/make-pod)))

(def ^{:doc " ecs-env expects an environment with the shape:
    {:aws {:creds  {:aws-access-key \"...\"
                    :aws-secret-key \"...\"
                    :endpoint \"eu-west-1\"}
           :ecs    {:cluster \"cluster-name\" :family \"task-name\"}
           :docker {:registry \"...\"
                    :api-version \"v1\"}}"}
    env
    (let [{:keys [creds]
           {:keys [registry api-version]} :docker
           {:keys [cluster family]} :ecs} (boot/get-env :aws)]
      {:creds creds
       :cluster cluster
       :family family
       :registry registry
       :version api-version
       :image family}))

(pod/with-eval-in pod
  (require '[uswitch.boot-containers.ecs :as ecs])
  (require '[uswitch.boot-containers.registry :as registry]))

(boot/deftask stop-tasks
  []
  (printf "Stopping tasks named \"%s\".\n" (:family ~env))
  (pod/with-eval-in pod
    (ecs/stop-tasks! ~env)))

(boot/deftask deploy
  [n number VAL int "Build no. to deploy, defaults to latest."
   t tag    VAL str "Tag to deploy, defaults to in-production."]
  (printf "Deploying build `n`=%s with deploy tag `t`=%s\n"
          (pr-str number)
          (pr-str tag))
  (pod/with-eval-in pod
    (let [[build id] (if ~number
                       (registry/image ~env ~number)
                       (first (registry/latest-builds ~env)))
          tag (or ~tag "in-production")]
      (printf "Tagging image \"%s\" with tag: \"%s\".\n" id tag)
      (registry/tag-image! ~env id tag)
      (printf "Stopping tasks named \"%s\"." (:family ~env))
      (println)
      (ecs/stop-tasks! ~env))))

(boot/deftask rollback
  [n number VAL int "Number of builds to rollback"
   t tag    VAL str "Tag to deploy, defaults to in-production."]
  (printf "Rolling back `n`=%s versions with deploy tag `t`=%s\n"
          (pr-str number)
          (pr-str tag))
  (pod/with-eval-in pod
    (let [number (or ~number 1)
          [build id] (nth (registry/latest-builds ~env) number)
          tag (or ~tag "in-production")]
      (printf "Tagging image \"%s\" with tag: \"%s\".\n" id tag)
      (registry/tag-image! ~env id tag)
      (printf "Stopping tasks named \"%s\"." (:family ~env))
      (println)
      (ecs/stop-tasks! ~env))))
