(ns uswitch.boot-containers.registry
  (:require
   [boot.core :as boot]
   [boot.pod :as pod]
   [clj-http.client :as http]
   [clojure.data.json :as json]))

(defn image-tags-url
  [{:keys [registry namespace version image] :or {namespace "library"}}]
  (format "https://%s/%s/repositories/%s/%s/tags"
          registry
          version
          namespace
          image))

(defn image-tags
  "Returns all tags for `{:keys [registry image] :as opts}`."
  [opts]
  (-> opts
      image-tags-url
      http/get
      :body
      (json/read-str :key-fn #(or (try (Integer. %) (catch Exception _)) %))))

(defn image
  "Return `[tag id]` with tag `tag`."
  [opts tag]
  [tag (-> (image-tags-url opts)
           (str \/ tag)
           (http/get)
           (:body)
           (json/read-str))])

(defn tag-image!
  "Tags image `image-id` with tag `tag`"
  [opts image-id tag]
  (let [url (str (image-tags-url opts) \/ tag)]
    (http/put url
              {:headers {"Content-Type" "application/json"}
               :body (pr-str image-id)})))

(defn latest-builds
  "Returns `([tag id] ...)` for numbered builds."
  [opts]
  (->> opts
       (image-tags)
       (filter (comp number? key))
       (sort-by key)
       (reverse)))

(defn build
  "Returns `[tag id]` for build number `n`."
  [opts n]
  (-> opts (image-tags) (select-keys [n]) (first)))
