(ns leiningen.unused-deps
  (:require [clojure.string :as str]
            [clojure.set :as set]
            [leiningen.core.project :as project]
            [leiningen.core.classpath :as classpath]
            [bultitude.core :as b]
            [clojure.java.io :as io])
  (:import (java.util.jar JarEntry JarFile)))

(defn used? [[_ contents] nses classes]
  (not (and (empty? (set/intersection nses (:nses contents)))
            (empty? (set/intersection classes (:classes contents))))))

(defn normalize [[d]]
  (if (= (namespace d) (name d))
    (symbol (name d))
    d))

(defn unused [{:keys [nses classes]} declared-dependencies]
  (map normalize (remove #(used? % nses classes) declared-dependencies)))

(defn expand-import [entry]
  (if (symbol? entry)
    [entry]
    (let [[prefix & classes] entry]
      (for [c classes]
        (symbol (format "%s.%s" prefix c))))))

(defn expand-require [entry]
  (if (symbol? entry)
    [entry]
    (first entry)))

(defn entry-for [kind entries]
  (case kind
    :use (map first entries)
    :require (map expand-require entries)
    :import (mapcat expand-import entries)
    []))

(defn ns-form->map [[_ _ & clauses]]
  ;; TODO: repeated require clauses are technically legal
  (into {} (for [[kind & entries] clauses]
             [kind (entry-for kind entries)])))

(defn used [paths]
  (let [files (map io/file (distinct paths))
        ns-forms (mapcat b/namespace-forms-in-dir files)
        ns-maps (map ns-form->map ns-forms)]
    {:nses (conj (set (mapcat :require ns-maps)) 'clojure.core)
     :classes (set (mapcat :import ns-maps))}))

(defn class-for [^JarEntry entry]
  (let [name (.getName entry)]
    (if (and (.endsWith name ".class") (not (re-find #"\$" name)))
      (-> name
          (str/replace ".class" "")
          (str/replace "/" ".")
          symbol))))

(defn classes-in [file]
  (with-open [jar (JarFile. file)]
    (set (keep class-for (enumeration-seq (.entries jar))))))

(defn classes-and-nses-for-resolved [file]
  (let [nses (set (map second (#'b/namespace-forms-in-jar file true)))]
    {:nses nses :classes (classes-in file)}))

(defn classes-and-nses [project]
  (into {} (for [d (keys (classpath/managed-dependency-hierarchy
                          :dependencies :managed-dependencies project))]
             [(first d) (classes-and-nses-for-resolved (:file (meta d)))])))

(defn find-unused-deps [project]
  (let [{:keys [source-paths test-paths]
         :as project} (project/unmerge-profiles project [:default])]
    (unused (used (concat source-paths test-paths))
            (classes-and-nses project))))

(defn unused-deps
  "Print a list of unused :dependencies declared in the project."
  [project]
  (doseq [u (find-unused-deps project)]
    (println u)))
