(ns uniui.react.core
  (:require [cljs.spec.alpha :as s]
            [uniui.react.react :as react]
            [uniui.react.react-dom :as react-dom]
            [uniui.dom :as dom]
            [uniui.core :as uniui])  )

(s/def ::react-element (s/or :dom dom/dom-element?
                             :component uniui/component-element?
                             :string string?
                             :string string?
                             :string symbol?
                             :number number?
                             :collection coll?))

(def ^:private component-registry (atom {}))

(defn- component->react-component
  [render]
  (fn
    [props]
    (let [cljs-props (js->clj props)]
      (react/create-fragment (render (:env props) (dissoc props :children :env) (:children props))))))

(defn- get-react-component
  [name render]
  (let [registry @component-registry]
    (if (some? (name registry))
      (name registry)
      (let [react-component (component->react-component render)]
        (swap! component-registry assoc name react-component)
        react-component))))

(defn render-element
  "Render a uniui element to a React element"
  [element env]
  (let [[element-type el] (s/conform ::react-element element)]
    (case element-type
      :dom (react/create-element (:name el)
                                 (-> el :props (dissoc :children) clj->js)
                                 (->> el :props :children (map #(render-element % env))))
      :component (react/create-element (get-react-component (:name el) (:render el))
                                       (-> el :props (dissoc :children) (assoc :env env))
                                       (->> el :props :children (or []) (map #(render-element % env))))
      :string (react/create-fragment [(name el)])
      :number (react/create-fragment [(str el)])
      :collection (react/create-fragment (map #(render-element % env) el)))))
(s/fdef render-element
        :args (s/cat :element ::react-element
                     :env map?)
        :ret react/valid-element?)

(defn render
  "Similar to ReactDOM.render; render an element to a DOM node"
  [element env node]
  (react-dom/render (render-element element env) node))
(s/fdef render
        :args (s/cat :element ::react-element
                     :env map?
                     :node #(instance? js/Node %))
        :ret any?)

(defn render-str
  "Similar to ReactDOMServer.render; render an element to an HTML string

  ```klipse
  (react/render-str (dom/h1 \"Hello, world!\") {})
  ```"
  [element env]
  (react-dom/render-to-static-markup (render-element element env)))
(s/fdef render
        :args (s/cat :element ::react-element
                     :env map?)
        :ret string?)