(ns tusk.core.types.large-object
  (:require
   [clojure.set :as set])
  (:import
   [org.postgresql.largeobject
    LargeObjectManager
    LargeObject]))

(defn ^:private
  modes
  [k]
  (or (get {:read       LargeObjectManager/READ
            :write      LargeObjectManager/WRITE
            :read-write LargeObjectManager/READWRITE
            :r          LargeObjectManager/READ
            :w          LargeObjectManager/WRITE
            :rw         LargeObjectManager/READWRITE}
           k)
      (throw (IllegalArgumentException.
              (str k " is not a valid mode")))))

(defn create
  ([lo-manager]
   (.createLO lo-manager))
  ([lo-manager mode]
   (.createLO lo-manager (modes mode))))

(defn open
  ([lo-manager oid]
   (open lo-manager oid nil))
  ([lo-manager oid {:keys [mode auto-commit?] :as opts}]
   (let [id       (long oid)
         selected (set/intersection #{:mode :auto-commit?}
                                    (set (keys opts)))
         m        (modes mode)]
     (condp = selected
       #{:mode :auto-commit?} (.open lo-manager id m auto-commit?)
       #{:auto-commit?}       (.open lo-manager id auto-commit?)
       #{:mode}               (.open lo-manager id m)
       nil)
     (.open lo-manager oid))))

(defn delete
  [lo-manager oid]
  (.delete lo-manager (long oid)))

(defn unlink
  [lo-manager oid]
  (.unlink lo-manager (long oid)))
