(ns tusk.core.types.geometric
  (:require
   [tusk.core.geometry :as g])
  (:import
   [org.postgresql.util
    PGobject]
   [org.postgresql.geometric
    PGpoint
    PGcircle
    PGbox
    PGline
    PGlseg
    PGpath
    PGpolygon]))

(defn pg-point
  [pt]
  (when-let [[x y] (g/point-2d pt)]
    (PGpoint. (double x) (double y))))

(defn pg-circle
  [c]
  (when-let [[[x y] r] (g/circle c)]
    (PGcircle. (double x) (double y) (double r))))

(defn pg-box
  [bx]
  (when-let [[[x1 y1] [x2 y2]] (g/box-2d bx)]
    (PGbox.
     (double x1) (double y1) (double x2) (double y2))))

(defn pg-line
  [l]
  (when-let [[[x1 y1] [x2 y2]] (g/line-2d l)]
    (PGline.
     (double x1) (double y1) (double x2) (double y2))))

(defn pg-lseg
  [l]
  (when-let [[[x1 y1] [x2 y2]] (g/line-segment-2d l)]
    (PGlseg.
     (double x1) (double y1) (double x2) (double y2))))

(defn pg-path
  ([l]
   (pg-path l false))
  ([l open?]
   (when-let [pts (->> (g/path-2d l)
                       (map pg-point)
                       (seq))]
     (PGpath. (into-array pts) (boolean open?)))))

(defn pg-polygon
  [ln]
  ;; postgres' polygon type isn't necessarily ring
  (when-let [pts (->> (g/path-2d ln)
                      (map pg-point)
                      (seq))]
    (PGpolygon. (into-array pts))))

(defn clj-vec
  [o]
  (condp instance? o
    PGpoint   [(.x o) (.y o)]
    PGcircle  [[(.x (.center o)) (.y (.center o))] (.radius o)]
    PGline    (let [x1 0
                    x2 10
                    y #(/ (- 0 (* % (.a o)) (.c o))
                          (.b o))]
                [[x1 (y x1)][x2 (y x2)]])
    PGbox     (mapv clj-vec (.point o))
    PGlseg    (mapv clj-vec (.point o))
    PGpath    (mapv clj-vec (.points o)) ;; closed/open missing
    PGpolygon (mapv clj-vec (.points o))
    nil))
