(ns tusk.util
  (:require
   [clojure.string :as s])
  (:import
   [org.postgresql.util
    PGobject]))

(defn parameter-name
  [stmt idx]
  (try
    (some-> stmt
            (.getParameterMetaData)
            (.getParameterTypeName idx))
    (catch Exception _ "")))

(defn set-hierarchy
  ([m]
   (doseq [[a b] m]
     (set-hierarchy a b)))
  ([p k]
   (when k
     (if (map? k)
       (do
         (set-hierarchy k)
         (doseq [[a _] k]
           (derive a p)))
       (derive k p)))))

(defn ^:private snake
  [s]
  (when (and s (string? s))
    (s/escape s {\- \_})))

(defn ^:private kebab
  [s]
  (when (and s (string? s))
    (s/escape s {\_ \-})))

(defn ident-str
  "convert from a clojure keyword to a postgres identifier.
  snake-case is kebab-cased and the schema becomes the key's namespace"
  [s]
  (when s
    (if (string? s)
      s
      (if-let [parents (when (keyword? s)
                         (some-> (namespace s)
                                 (s/split #"\.")))]
        (-> (namespace s)
            (s/split #"\.")
            (conj (name s))
            (->> (map #(str \" % \"))
                 (map snake)
                 (s/join ".")))
        (name s)))))

(defn ident-key
  "convert from a clojure keyword to a postgres identifier.
  kebab-case is snake-cased and the key's namesace becomes the schema"
  [k]
  (when k
    (if (keyword? k)
      k
      (if-let [path (->> (re-seq #"\"([^\"]+)\"\.?" k)
                         (map second)
                         seq)]
        (keyword (->> path butlast (map kebab) (s/join \.))
                 (->> path last kebab))
        (keyword k)))))

(defn pg-object
  "turn a value into a postgres object of a specified type"
  [pg-type v]
  {:pre [(or (string? pg-type)
             (keyword? pg-type))]}
  (when v
    (let [typ (ident-str pg-type)
          val (cond
                (keyword? v) (name v)
                (string? v)  v
                :else        (str v))] ;; this won't happen
      (doto (PGobject.)
        (.setType typ)
        (.setValue val)))))
