(ns tusk.types.geometric
  (:require
   [tusk.geometry :as g]
   [tusk.core :as c]
   [tusk.types.clojure :as clj])
  (:import
   [org.postgresql.util
    PGobject]
   [org.postgresql.geometric
    PGpoint
    PGcircle
    PGbox
    PGline
    PGlseg
    PGpath
    PGpolygon]))

(defn pg-point
  [pt]
  (when-let [[x y] (g/point-2d pt)]
    (PGpoint. (double x) (double y))))

(defn pg-circle
  [c]
  (when-let [[[x y] r] (g/circle c)]
    (PGcircle. (double x) (double y) (double r))))

(defn pg-box
  [bx]
  (when-let [[[x1 y1] [x2 y2]] (g/box-2d bx)]
    (PGbox.
     (double x1) (double y1) (double x2) (double y2))))

(defn pg-line
  [l]
  (when-let [[[x1 y1] [x2 y2]] (g/line-2d l)]
    (PGline.
     (double x1) (double y1) (double x2) (double y2))))

(defn pg-lseg
  [l]
  (when-let [[[x1 y1] [x2 y2]] (g/line-segment-2d l)]
    (PGlseg.
     (double x1) (double y1) (double x2) (double y2))))

(defn pg-path
  ([l]
   (pg-path l false))
  ([l open?]
   (when-let [pts (->> (g/path-2d l)
                       (map pg-point)
                       (seq))]
     (PGpath. (into-array pts) (boolean open?)))))

(defn pg-polygon
  [ln]
  ;; postgres' polygon type isn't necessarily ring
  (when-let [pts (->> (g/path-2d ln)
                      (map pg-point)
                      (seq))]
    (PGpolygon. (into-array pts))))

(defn clj-vec
  [o]
  (condp instance? o
    PGpoint   [(.x o) (.y o)]
    PGcircle  [[(.x (.center o)) (.y (.center o))] (.radius o)]
    PGline    (let [x1 0
                    x2 10
                    y #(/ (- 0 (* % (.a o)) (.c o))
                          (.b o))]
                [[x1 (y x1)][x2 (y x2)]])
    PGbox     (mapv clj-vec (.point o))
    PGlseg    (mapv clj-vec (.point o))
    PGpath    (mapv clj-vec (.points o))
    PGpolygon (mapv clj-vec (.points o))
    nil))

(defn geom-edn
  "clojure map representing the geometric type

  geo-json : geographic/json
  geom-edn : geometry/edn"
  [o]
  (some->
   (condp instance? o
     PGpoint   {:type :point}
     PGcircle  {:type :circle}
     PGline    {:type :line}
     PGbox     {:type :box}
     PGlseg    {:type :line-segment}
     PGpath    {:type       :path
                :properties {:open? (.isOpen o)}}
     PGpolygon {:type :polygon}
     nil)
   (assoc :coordinates (clj-vec o))))

(defmethod c/coerce [::clj/sequential :point]   [v _] (pg-point v))
(defmethod c/coerce [::clj/sequential :circle]  [v _] (pg-circle v))
(defmethod c/coerce [::clj/sequential :box]     [v _] (pg-box v))
(defmethod c/coerce [::clj/sequential :line]    [v _] (pg-line v))
(defmethod c/coerce [::clj/sequential :lseg]    [v _] (pg-lseg v))
(defmethod c/coerce [::clj/sequential :path]    [v _] (pg-path v))
(defmethod c/coerce [::clj/sequential :polygon] [v _] (pg-polygon v))
(defmethod c/coerce [::clj/map        :point]   [v _] (pg-point (:coordinates v)))
(defmethod c/coerce [::clj/map        :circle]  [v _] (pg-circle (:coordinates v)))
(defmethod c/coerce [::clj/map        :box]     [v _] (pg-box (:coordinates v)))
(defmethod c/coerce [::clj/map        :line]    [v _] (pg-line (:coordinates v)))
(defmethod c/coerce [::clj/map        :lseg]    [v _] (pg-lseg (:coordinates v)))
(defmethod c/coerce [::clj/map        :path]    [v _] (pg-path (:coordinates v) (-> v :properties :open?)))
(defmethod c/coerce [::clj/map        :polygon] [v _] (pg-polygon (:coordinates v)))
