(ns tusk.types.range
  (:require
   [tusk.core :as c]
   [tusk.types :as t]
   [tusk.types.clojure :as clj]))

(defn read-with
  "takes a string representing a postgres range
  and returns a map with the keys"
  [val-fn]
  (fn [rng]
    (let [inclusive?    #(condp = %
                           "(" false
                           ")" false
                           "[" true
                           "]" true
                           nil)
          parse         #(when-not (empty? %)
                           (val-fn %))
          range         #"(\[|\()([^,]*),([^,]*)(\]|\))"
          [_ l? l u u?] (re-matches range rng)
          lower         (parse l)
          upper         (parse u)
          lower?        (inclusive? l?)
          upper?        (inclusive? u?)]
      {:upper-inclusive? upper?
       :lower-inclusive? lower?
       :lower lower
       :upper upper})))

(defn write-with
  [str-fn]
  (fn [{:keys [lower upper] :as r}]
    (let [[l-in l-ex u-ex u-in] ["[" "(" ")" "]"]
          l? (get r :lower-inclusive? true)
          u? (get r :upper-inclusive? false)]
      (format "%s%s,%s%s"
              (if l? l-in l-ex)
              (if lower (str-fn lower) "")
              (if upper (str-fn upper) "")
              (if u? u-in u-ex)))))

(defmacro def-pg-range
  [type-name coerce-fn]
  `(let [read#  (read-with  ~coerce-fn)
         write# (write-with ~coerce-fn)]
     (defmethod c/coerce [::clj/map  ~type-name]       [v# _#] (write# v#))
     (defmethod c/coerce [~type-name ::clj/collection] [v# _#] (read#  v#))))

(def-pg-range :daterange t/sql-date)
(def-pg-range :tstzrange t/sql-tstz)
(def-pg-range :int4range t/sql-int4)
(def-pg-range :int8range t/sql-int8)
(def-pg-range :numrange  t/sql-numeric)
