(ns tusk.types.postgis
  (:require
   [tusk.util :as u]
   [tusk.geometry :as g])
  (:import
   [org.postgresql.util
    PGobject]
   [org.postgis
    ;; PGbox3d
    PGbox2d
    Point
    MultiPoint
    LineString
    LinearRing
    MultiLineString
    Polygon
    MultiPolygon
    PGgeometryLW
    PGgeometry
    Geometry]))

(defn pg-point2d
  [p]
  (when-let [[x y] (g/point-2d p)]
    (Point. (double x) (double y))))

(defn pg-box2d
  [bx]
  (when-let [[[left lower]
              [right upper]] (g/box-2d bx)]
    (PGbox2d.
     (Point. (double left) (double lower))
     (Point. (double right) (double upper)))))

(defn pg-multi-point2d
  [points]
  (when-let [p (g/path-2d points)]
    (MultiPoint.
     (into-array Point (map pg-point2d p)))))

(defn pg-line-string2d
  [points]
  (when-let [p (g/path-2d points)]
    (LineString.
     (into-array Point (map pg-point2d p)))))

(defn pg-multi-line-string2d
  [paths]
  (when-let [p (g/multipath-2d paths)]
    (MultiLineString.
     (into-array LineString (map pg-line-string2d p)))))

(defn pg-linear-ring2d
  [points]
  (when-let [p (g/ring-2d points)]
    (LinearRing.
     (into-array Point (map pg-point2d p)))))

(defn pg-polygon2d
  [rings]
  (when-let [r (g/multiring-2d rings)]
    (Polygon.
     (into-array LinearRing (map pg-linear-ring2d r)))))

(defn pg-multi-polygon2d
  [polygons]
  (when-let [p (map g/multiring-2d polygons)]
    (when (every? (complement nil?) p)
      (MultiPolygon.
       (into-array Polygon (map pg-polygon2d polygons))))))

(defn pg-geometry
  [g]
  (when (instance? Geometry g)
    (PGgeometryLW. g)))

(defn set-srid
  [g s]
  (when (instance? Geometry g)
    (doto g (.setSrid s))))

(defn clj-vec
  [o]
  (condp instance? o
    PGgeometry      (clj-vec (.getGeometry o))
    PGgeometryLW    (clj-vec (.getGeometry o))
    Point           (if (= (.dimension o) 3)
                      [(.x o) (.y o) (.z o)]
                      [(.x o) (.y o)])
    MultiPoint      (mapv clj-vec (.getPoints o))
    LineString      (mapv clj-vec (.getPoints o))
    MultiLineString (mapv clj-vec (.getLines o))
    LinearRing      (mapv clj-vec (.getPoints o))
    Polygon         (mapv clj-vec (for [i (range (.numRings o))]
                                    (.getRing o i)))
    MultiPolygon    (mapv clj-vec (.getPolygons o))
    nil))

(defn geo-edn
  "clojure map representing the geographic type"
  [o]
  (let [geo-edn* (fn [t]
                   {:type        t
                    ;; :properties {:srid (.getSrid o)
                    ;;              :dimensions (.getDimension o)}
                    :coordinates (clj-vec o)})]
    (condp instance? o
      PGgeometry      (geo-edn (.getGeometry o))
      PGgeometryLW    (geo-edn (.getGeometry o))
      Point           (geo-edn* :point)
      MultiPoint      (geo-edn* :multi-point)
      LineString      (geo-edn* :line-string)
      MultiLineString (geo-edn* :multi-line-string)
      LinearRing      (geo-edn* :linear-ring)
      Polygon         (geo-edn* :polygon)
      MultiPolygon    (geo-edn* :multi-polygon)
      nil)))
