(ns tusk.core.geometry
  (:require
   [tusk.core.util :refer [hobs]]
   [clojure.set :as set]))

(defn ^:private not-nil?
  [x]
  (not (nil? x)))

(defn ^:private sequential
  [x]
  (when (sequential? x)
    x))

(defn point-2d
  [pt]
  (when-let [[x y] (sequential pt)]
    (when (and (number? x)
                (number? y)
                (= 2 (count pt)))
      [x y])))

(defn point-3d
  [pt]
  (when-let [[x y z] (sequential pt)]
    (when (and (number? x)
               (number? y)
               (number? z)
               (= 3 (count pt)))
      [x y z])))

(defn circle
  [c]
  (when-let [[xy r] (sequential c)]
    (when (and (point-2d xy)
               (and (number? r) (pos? r))
               (= 2 (count c)))
      [(point-2d xy) r])))

(defn sphere
  [c]
  (when-let [[xy r] (sequential c)]
    (when (and (point-3d xy)
               (and (number? r) (pos? r))
               (= 2 (count c)))
      [(point-3d xy) r])))

(defn box-2d
  [bx]
  (let [[p1 p2 :as pts] (sequential bx)
        [x1 y1]         (point-2d p1)
        [x2 y2]         (point-2d p2)]
    (when (and x1 y1 x2 y2 (= 2 (count pts)))
      (let [left  (min x1 x2)
            right (max x1 x2)
            upper (max y1 y2)
            lower (min y1 y2)]
        [[right upper] [left lower]]))))

(defn box-3d
  [bx]
  (let [[p1 p2 :as pts] (sequential bx)
        [x1 y1 z1]      (point-3d p1)
        [x2 y2 z2]      (point-3d p2)]
    (when (and x1 y1 x2 y2 (= 2 (count pts)))
      (let [left   (min x1 x2)
            right  (max x1 x2)
            upper  (max y1 y2)
            lower  (min y1 y2)
            top    (max z1 z2)
            bottom (min z1 z2)]
        [[right upper] [left lower]]))))

(defn ^:private entirely
  [sub-type]
  (fn [c]
    (when-let [v (some->> (sequential c)
                          (mapv sub-type))]
      (when (every? not-nil? v)
        v))))

(defn ^:private meets
  [pred]
  (fn [v]
    (when (and (sequential? v)
               (pred v))
      v)))

(defn ^:private longer-than
  [ct]
  (fn [v]
    (when (and (sequential? v)
               (> (count v) ct))
      v)))

(defn ^:private length-of
  [ct]
  (fn [v]
    (when (and (sequential? v)
               (= (count v) ct))
      v)))


(def line-2d (comp
              (length-of 2)
              (entirely point-2d)))
(def line-3d (comp
              (length-of 2)
              (entirely point-3d)))

(def line-segment-2d (comp
                      (length-of 2)
                      (entirely point-2d)))
(def line-segment-3d (comp
                      (length-of 2)
                      (entirely point-3d)))

;; TODO: add self crossing check
(def path-2d (comp
              (longer-than 1)
              (entirely point-2d)))
;; TODO: add self crossing check
(def path-3d (comp
              (longer-than 1)
              (entirely point-3d)))

(def multipoint-2d (comp
                    (longer-than 0)
                    (entirely point-2d)))

(def multipoint-3d (comp
                    (longer-than 0)
                    (entirely point-3d)))

(defn ^:private first=last
  [x]
  (= (first x) (last x)))

;; TODO: add self crossing check
(def ring-2d (comp
              (meets first=last)
              (longer-than 3)
              (entirely point-2d)))
;; TODO: add self crossing check
(def ring-3d (comp
              (meets first=last)
              (longer-than 3)
              (entirely point-3d)))

(def multiring-2d (comp
                   (longer-than 0)
                   (entirely ring-2d)))

(def multiring-3d (comp
                   (longer-than 0)
                   (entirely ring-3d)))

(def multipath-2d (comp
                   (longer-than 0)
                   (entirely path-2d)))

(def multipath-3d (comp
                   (longer-than 0)
                   (entirely path-3d)))

(defn link
  [c]
  (if (not= (first c) (last c))
    (conj (vec c) (last c))
    c))

(defn unlink
  [c]
  (if (= (first c) (last c))
    (vec (butlast c))
    c))

(defn identify
  [c]
  (if-not (sequential? c)
    (cond
      (number? c) c
      :else       nil)
    (let [coords     (vec c)
          kids       (mapv identify coords)
          point?     (and (every? number? coords)
                          (< 1 (count coords)))
          kids-are?  (fn [t]
                       (every? #(-> % :types (contains? t))
                               kids))
          add-types  (fn [m & ts]
                       (update m :types set/union (set ts)))
          points?    (kids-are? :point)
          ring?      (and points?
                          (> (count coords) 2)
                          (= (first coords) (last coords)))
          box?       (and points? (= 2 coords))
          next-depth #(when-let [x (seq (map :depth kids))]
                        (inc (apply max x)))
          dimensions (apply set/union (map :dimensions kids))
          complex?   (and kids (every? map? kids))]
      (cond
        point?        {:types      #{:point}
                       :dimensions #{(count coords)}
                       :depth      0
                       :coords     coords}
        complex?      (cond-> {:types      #{}
                               :dimensions dimensions
                               :children   kids
                               :coords     coords}
                        ring?                (add-types :ring)
                        box?                 (add-types :box)
                        (kids-are? :point)   (add-types :multipoint :path)
                        (kids-are? :path)    (add-types :multipath)
                        (kids-are? :ring)    (add-types :polygon)
                        (kids-are? :polygon) (add-types :multipolygon)
                        :always              (-> (assoc :depth (next-depth))
                                                 (add-types :collection)))
        (some nil?
              coords) nil
        :else         nil))))

(defn ^:private dimensions
  [d]
  (fn [id]
    (when (= (:dimensions id) #{d})
      id)))

(defn ^:private structure
  [k]
  (fn [id]
    (when (-> id :types (contains? k))
      id)))


;; BROKEN !!!!!
#_(defn bounding-box
;; BROKEN !!!!!
  ([] nil)
  ([p1 & [p2 & after-p2 :as more]]
   (cond
     (not (point-2d p1))   (box-2d (first p1) (rest p1) more)
     (and (point-2d p1)
          (point-2d p2))   (let [[x1 y1] p1
                                 [x2 y2] p2
                                 left    (min x1 x2)
                                 right   (max x1 x2)
                                 upper   (max y1 y2)
                                 lower   (min y1 y2)]
                             (if (seq? after-p2)
                               (box-2d [[right upper] [left lower]] after-p2)
                               [[right upper] [left lower]]))
     (not (or (vector? p2)
              (list? p2))) nil
     (empty? p2)           (box-2d p1 more)
     :else                 (box-2d p1 (first p2) (rest p2) more))))
;; BROKEN !!!!!
