(ns truckerpath.clj-config.core
  (:require [clojure.edn :as edn]
            [clojure.tools.logging :as log])
  (:refer-clojure :exclude [reset! get get-in]))

(defn read-edn
  "Reads file and parse it assuming it's EDN format."
  [filename]
  (log/info "Reading EDN file" filename)
  (edn/read-string (slurp filename)))

(defn load!
  "Reads EDN file and store it in app-config."
  [cfg filename]
  (log/info "Loading config from" filename)
  (let [new-cfg (read-edn filename)]
    (clojure.core/reset! cfg new-cfg)))

(defn reset!
  "Resets app-config to a given one for testing and more complex operations
   (like merging two configs)."
  [cfg new-cfg]
  (log/info "Reseting config")
  (clojure.core/reset! cfg new-cfg))

(defn get-in
  "Returns the value in a nested config. Return nil if the key is not present."
  [cfg [& keys]]
  (if @cfg
    (let [not-found ::not-found
          value (clojure.core/get-in @cfg keys not-found)]
      (if (not= value not-found)
        value
        (log/error "Unable to get value for" keys "- config doesn't have such key")))
    (log/error "Unable to get value for" keys "- config isn't initialized yet.")))

(defn get
  "Returns the value in config. Return nil if the key is not present."
  [cfg key]
  (get-in cfg [key]))

;; (defn config
;;   "Returns whole config."
;;   []
;;   @app-config)
