(ns truckerpath.anti-flooding-http.core
  (:require [clj-http.client :as http]
            [clojure.tools.logging :as log])
  (:refer-clojure :exclude [get]))

(def active-requests (ref {}))

(defn- map-key
  "Make a key for request to be stored in a hash-map"
  [method url]
  (str method " " url))

(defn- do-request!
  "Make an actual http request and delete request from
   map of active aftewards"
  [method-fn method url opts]
  (log/debug "Requesting" method url)
  (try
    (method-fn url opts)
    (finally (dosync
              (log/debug "Deleting" method url)
              (alter active-requests dissoc (map-key method url))))))

(defn- lazy-http!
  "Make an http request if there's no the same (method & url) request
   waiting for response at this moment. It helps to avoid flooding
   servers with identical requests.
   If there're no identical requests, then function makes a request,
   stores a delay of this http-request into a map of active requests and
   blocks until it's done.
   If there's already such request - just takes stored delay and blocks
   until it's done."
  [method-fn method url opts]
  (if-let [existing-request (@active-requests (map-key method url))]
    (do
      (log/debug "Waiting for" method url)
      @existing-request)

    ; Dereferencing only after transaction successfully completion
    ; to avoid side-effects (http requests) duplication
    @(dosync
      ; Double check for existing request in transaction
      ; Otherwise duplicated request might be fired
      (if-let [existing-request (@active-requests (map-key method url))]
        (do
          (log/debug "Waiting for" method url)
          existing-request)
        (let [new-request (delay (do-request! method-fn method url opts))]
          (alter active-requests assoc (map-key method url) new-request)
          new-request)))))

(defn head
  "HEAD request with anti-flooding magic. See `lazy-http!` for more details."
  [url & [opts]]
  (lazy-http! http/head "HEAD" url (or opts {})))

(defn get
  "GET request with anti-flooding magic. See `lazy-http!` for more details."
  [url & [opts]]
  (lazy-http! http/get "GET" url (or opts {})))

(defn post
  "Just a wrapper around `clj-http.client/post` for now."
  [url & [opts]]
  (http/post url (or opts {})))

(defn put
  "Just a wrapper around `clj-http.client/put` for now."
  [url & [opts]]
  (http/put url (or opts {})))

(defn patch
  "Just a wrapper around `clj-http.client/patch` for now."
  [url & [opts]]
  (http/patch url (or opts {})))

(defn delete
  "Just a wrapper around `clj-http.client/delete` for now."
  [url & [opts]]
  (http/delete url (or opts {})))
