(ns clj-web.route
  "Router module. Determines which, if any, function should be called
   given an HTTP request.

   Router configuration is specified by the route-map, which is expected
   to be of the following form:

       {PATTERN {METHOD [HANDLER DATA]
                 ...}
        ...}

   PATTERN - String: URL path pattern to match. Uses the syntax defined
             by Clout (https://github.com/weavejester/clout).
   METHOD  - Symbol: HTTP method, as uppercase symbol.
   HANDLER - Symbol: Fully-qualified name of the function that handles the
             route. The handler should accept a single argument, which is the
             Ring Request Map.
   DATA    - (Optional) Any: Additional data to be passed to the handler when
             the route is called. Will be passed in the request map under the
             key :route-data.
  "
  (:require [clj-web.util :refer [p ring-response-map? vec*]]
            [clj-web.response :refer [method-not-allowed not-found]]
            [clojure.string :refer [lower-case]]
            [clout.core :as clout]))

(defn- route-reducer [req _ route]
  (let [[compiled-route methods] route]
    (when-let [match (clout/route-matches compiled-route req)]
      (reduced
       (let [[handler data] (get methods (:request-method req))
             req            (assoc req :route-params match
                                       :route-data   data)]
         (if handler
           [handler req]
           [(fn [_] (method-not-allowed)) req]))))))

(defn- router [compiled-routes req]
  (if-let [result (reduce (p route-reducer req) nil compiled-routes)]
    result
    [(fn [_] (not-found)) req]))

(defn- resolve-handler [f]
  (let [ns (symbol (namespace f))]
    (require ns)
    (ns-resolve ns (symbol (name f)))))

(defn- compile-route-targets-map [targets]
  (into
   {}
   (for [[method slug] targets
         :let [[f data] (vec* slug)]]
     [(-> method str lower-case keyword)
      [(resolve-handler f) data]])))

(defn- compile-route-map [route-map]
  (into
   {}
   (for [[path targets] route-map]
     [(clout/route-compile path)
      (compile-route-targets-map targets)])))

(defn make-router
  "This function compiles route-map into a function which can then be called
   to route an HTTP request and call the appropriate handler. The returned
   function takes a single argument, the Ring request map, and returns a Ring
   response map."
  [route-map]
  {:pre [(map? route-map)]}
  (p router (compile-route-map route-map)))
