(ns clj-web.hook
  "Hook module. This implements clj-web's notion of hooks and hook chains.

   The hook chain is a method of reducing a piece of state over a series of
   functions. There are two key differences between this and an ordinary
   reduction: automatic wrapping and unwrapping of arguments/returns, and
   a mechanism to restrict a hook's ability to overwrite state.

   The goal is to provide a (more traditional, perhaps) alternative to the
   middleware pattern that is easier to write and reason about.

   Each hook should be a function of a single argument, where it will be
   passed the current state. The hook chain will inspect the hook's return
   value to determine if the hook was sucessful, or if the chain should be
   aborted.

   For the hook to be considered a failure, the response must be a map and must
   contain the key :clj-web.hook/status with value set to :clj-web.hook/abort.
   Any other return value will be considered a success. If the response is a
   success, the chain will continue; if the response is a failure, the chain
   will stop, returning the value specified by the key :clj-web.hook/result.")

(defn abort
  "Convenience function for constructing an hook abort response."
  [result]
  {::status ::abort
   ::result result})

(defn aborted?
  "Predicate whether a value should be considered a hook abort response."
  [result]
  (and (map? result) (= (::status result) ::abort)))

(defn chain*
  "Run a hook chain with arbitrary specificiation.
   You probably don't want to call this directly."
  [config state hooks]
  (let [{unwrap-hook :unwrap
         wrap-result :wrap} config]
    (reduce
     (fn [state hook]
       (let [hook-fn (unwrap-hook hook)
             result  (hook-fn state)]
         (if (aborted? result)
           (reduced result)
           (wrap-result hook state result))))
     state
     hooks)))

(defn chain
  "Basic hook chain. Hooks are a simple sequence of hook functions. Each
   successful hook response will reset the state for the next.

   Example:

       (chain {:v 1}
              [(fn [state]
                 {:v 2})
               (fn [state]
                 (if (= (:v state) :foobar)
                   (abort {:v -1})
                   state))
               (fn [state]
                 {:v (* (:v state) 2)})])
       => {:v 4}
  "
  [state hooks]
  (chain* {:unwrap identity
           :wrap   (fn [_ _ x] x)}
          state
          hooks))

(defn chain-keyed
  "Keyed hook chain: each hook's response assoc-ed into the state under
   the specified key.

   Example:

       (chain-keyed {:v 1}
                    [:x #(* 2 (:v %))
                     :y #(* 3 (:x %))
                     :z (fn [state]
                          (if (= (:y state) 20)
                            (abort :foobar)
                            42))])
       => {:v 1 :x 2 :y 6 :z 42}
  "
  [state hooks]
  {:pre [(even? (count hooks))]}
  (chain* {:unwrap second
           :wrap   (fn [h s x] (assoc s (first h) x))}
          state
          (partition-all 2 hooks)))
