(ns clj-web.core
  "clj-web: a library to help you write web apps

   This module is the main entry point to clj-web.


   Request processing pipeline:

            Request
               ↓
       +--------------+
       | *Prerouting* |
       +--------------+
              ↓
          +--------+
          | Router |
          +--------+
              ↓
       +---------------+
       | *Postrouting* |
       +---------------+
              ↓
         +---------+
         | Handler |
         +---------+
              ↓
        +------------+
        | *Response* |
        +------------+
              ↓
           Response


   ** Any stage named with earmuffs represents a hook chain.
  "
  (:require [clj-web.hook :as hook]
            [clj-web.route :as route]
            [clojure.edn :as edn]
            [clojure.java.io :as io]))

(defn- load-route-map
  "The route-map may be passed either directly or encoded into an
   edn-formatted file. If the argument is a string, it will be interpreted as
   the name of a Java resource, which will then be read, and the contents used
   as the route-map."
  [rm]
  (cond
    (map? rm) rm
    (string? rm) (edn/read-string (slurp (io/resource rm)))
    :else (throw (Exception. (str "Route-map must be either a map or the "
                                  "string name of an edn-formatted "
                                  "resource file.")))))

(def ^:private init-chain
  "Internal hook chain. Preprocesses the request. Runs before everything else."
  [
   ])

(defn handler
  "This is the main entry point to clj-web. This function returns a handler
   function representing your web app. Pass the returned function to a Ring-
   compatible web server to run the app.

   The app is configured through the map passed as this function's argument:

       :route-map   - (Map|String): Defines the app's routes. May be either the
                        route-map itself, or a string naming a Java resource
                        that defines the route-map in edn format. See
                        clj-web.route for details on the route-map format.
       :prerouting  - Vector: Keyed hook chain, processes the request. Runs
                        prior to the router. See clj-web.hook for details on
                        hook format.
       :postrouting - Vector: Keyed hook chain, processes the request. Runs
                        after the router but prior to the request handler. See
                        clj-web.hook for details on hook format.
       :response    - Vector: Hook chain, processes the response. Runs after the
                        request handler but prior to sending the response. See
                        clj-web.hook for details on hook format.
  "
  [& {:keys [route-map prerouting postrouting response]}]
  (let [router (route/make-router (load-route-map route-map))]
    (fn [req]
      (let [req (hook/chain-keyed req init-chain)]
        (if (hook/aborted? req)
          (::hook/result req)
          (let [req (hook/chain-keyed req prerouting)]
            (if (hook/aborted? req)
              (::hook/result req)
              (let [[handler req] (router req)
                    req           (hook/chain-keyed req postrouting)]
                (if (hook/aborted? req)
                  (::hook/result req)
                  (let [resp (hook/chain (handler req) response)]
                    (if (hook/aborted? resp)
                      (::hook/result resp)
                      resp)))))))))))
