(ns toy-app.utility
  (:use
    [clj-time.core :exclude [extend]]
    [clj-time.coerce]
    [clj-time.format]
    [clojure.java.io]
   )
  (:import java.util.Locale )
  )

; a collection of general methods with no dependecies, so this can be included anywhere
;
;

(defn slurp-split [f] (clojure.string/split (slurp f) #"\n"  ))

(defn readlines [f]
  (with-open [r (reader f)]
    (doall (line-seq r))))

;(defn try-nil
;  ([body] (try-nil body Exception))
;  ([body e] (try body (catch e nil))))



(defn memoize-ttl
  "Like `memoize` but invalidates the cache for a set of arguments after TTL
  msecs has elapsed."
  [ttl f]
  (let  [cache  (atom  {})]
    (fn  [& args]
      (let  [{:keys [time-cached d-result]}  (@cache args)
             now (System/currentTimeMillis)]

        (if (and time-cached (< (- now time-cached) ttl))
          @d-result
          (let [d-result (delay (apply f args))]
            (swap! cache assoc args  {:time-cached now :d-result d-result})
            @d-result))))))

(defn in?
  "true if seq contains elm"
  [seq elm]
  (some #(= elm %) seq))


(def twitter-formatter (formatter "EEE MMM dd HH:mm:ss Z yyyy"))

(defn nothing?
  "true if nil or empty string or empty collection"
  [x]
  (or
    (nil? x)
    (and
      (string? x)
      (clojure.string/blank? x))
    (and
      (coll? x)
      (empty? x))))

(defn twitter_to_ts [ts]
  (parse (with-locale twitter-formatter Locale/ENGLISH) ts ))
(defn twitter_to_unix [ts]
  (to-long (twitter_to_ts ts)))

(defn get_hour [ts_object] (hour ts_object))


(defn get_week [ts_object] (Integer. (unparse (formatter "w") ts_object)))

(defn long_to_time [ts] (from-long ts))

(defn extract-map [n coll]
  (into {}
    (map
      (fn [[k v]] [(last (re-find (re-pattern (str  n "\\[(\\S+)\\]")) k)) v]) 
      (filter #(re-find (re-pattern (str n "\\[(\\S+)\\]")) (name (first %)) ) coll))))


(defn take-more [n coll]
  (take n (concat coll (repeat nil))))

;http://stuartsierra.com/2011/01/09/edit-quotient
;
(defn levenshtein [s t]
  (let [m (count s)
        n (count t)
        d (make-array Integer/TYPE (inc m) (inc n))]
    (dotimes [i m] (aset-int d i 0 i))
    (dotimes [j n] (aset-int d 0 j j))
    (doseq [j (range 1 (inc n))]
      (doseq [i (range 1 (inc m))]
        (if (= (.charAt s (dec i)) (.charAt t (dec j)))
          (aset-int d i j (aget d (dec i) (dec j)))
          (aset-int d i j (min (inc (aget d (dec i) j))
                               (inc (aget d i (dec j)))
                               (inc (aget d (dec i) (dec j))))))))
    (aget d m n)))


(defn levenshtein-quotient [s t]
  (let [sum (+ (count s) (count t))]
    (if (pos? sum)
      (/ (levenshtein s t)
         (/ sum 2))
      0)))

;https://gist.github.com/eliasson/1302024
(defn md5-hash
  "Generate a md5 checksum for the given string"
  [token]
  (let [hash-bytes
         (doto (java.security.MessageDigest/getInstance "MD5")
               (.reset)
               (.update (.getBytes token)))]
       (.toString
         (new java.math.BigInteger 1 (.digest hash-bytes)) ; Positive and the size of the number
         16)))

(defmacro tt-log
  "Evaluates expr and logs the time it took. Returns the value of expr.
   e.g. (tt-log log/err \"time for func-a\" (func-a ...))"
  [log-func tag expr]
  `(let [start# (System/nanoTime)
         ret# ~expr]
     (~log-func
      (str ~tag " time: "
           (/ (double (- (System/nanoTime) start#)) 1000000.0)
           " msecs"))
     ret#))

(defmacro tt
  "Evaluates expr and print the time it took. Returns the value of expr.
   e.g. (tt \"time for func-a\" (func-a ...))"
  [tag expr]
  `(tt-log println ~tag ~expr))

(def ^{:private true} email-formatter #"^[a-zA-Z0-9._%+-]+@(?:[a-zA-Z0-9-]+\.)+(?:[a-zA-Z]{2}|com|org|net|edu|gov|mil|biz|info|mobi|name|aero|asia|jobs|museum)$")

(defn check-email-format [email-address]
  (re-matches email-formatter email-address))

(defonce ^{:private true} default-bad-words-set
  (-> (with-open [in (clojure.java.io/reader
                      (clojure.java.io/resource "bad-words.txt"))]
        (doall
         (filter seq (map clojure.string/trim (line-seq in)))))
      set))
