(ns tidy.core
  (:require [clojure.core.async :as core-async]
            [clojure.main :refer [repl-read]]
            [utilis.fn :refer [fsafe]]
            [utilis.map :refer [compact map-vals map-keys]]
            [clojure.set :refer [difference]]
            [clojure.string :as st]
            [tidy.core :as tidy])
  (:import [java.util.concurrent LinkedBlockingQueue]))

;;; Declarations

(declare run* dispose* subscribe* unsubscribe* deref* stabilized?)

(def ^:dynamic *deref-context* nil)
(def ^:private run-opt-keys [:silent? :best-effort?])

(def prn-monitor (Object.))
(defn prn*
  [& args]
  (locking prn-monitor
    (apply prn args)))

(def global-monitor (Object.))

(defprotocol IReaction
  (dispose [this])
  (disposed? [this])
  (run [this] [this opts]))

(defprotocol ISubscribable
  (subscribe
    [this]
    [this opts]
    [this key opts])
  (unsubscribe [this key]))

(defprotocol IInitializable
  (initialized? [this]))

(defprotocol IReactiveAtom)

;;; Records

(defrecord RAtom [value]
  IReactiveAtom

  IInitializable
  (initialized? [this]
    (not= :tidy/none @value))

  clojure.lang.IDeref
  (deref [this] (deref* this))

  clojure.lang.IAtom
  (reset [this new-value] (.reset ^clojure.lang.IAtom value new-value))
  (swap [this f]          (.swap ^clojure.lang.IAtom value f))
  (swap [this f x]        (.swap ^clojure.lang.IAtom value f x))
  (swap [this f x y]      (.swap ^clojure.lang.IAtom value f x y))
  (swap [this f x y more] (.swap ^clojure.lang.IAtom value x y more))

  clojure.lang.IRef
  (addWatch [this key f]  (.addWatch ^clojure.lang.IAtom value key f) this)
  (removeWatch [this key] (.removeWatch ^clojure.lang.IAtom value key) this)
  (getWatches [this]      (.getWatches value))

  Object
  (toString [this] (str @this)))

(defn ratom?
  [x]
  (satisfies? IReactiveAtom x))

(defn ratom
  ([] (ratom :tidy/none))
  ([initial-state] (RAtom. (atom initial-state))))

(defrecord Reaction [computation-fn
                     value state
                     listeners
                     inputs
                     ref-count
                     value-queue]
  IReaction
  (dispose [this]   (locking global-monitor (dispose* this)))
  (disposed? [this] (= :tidy/disposed @state))
  (run [this]       (locking global-monitor (run* this nil)))
  (run [this opts]  (locking global-monitor (run* this opts)))

  IInitializable
  (initialized? [this]
    (and (= :tidy/started @state)
         (not= :tidy/none @value)
         (every? initialized? @inputs)))

  ISubscribable
  (subscribe [this key opts]
    (locking global-monitor
      (subscribe* this key opts)))
  (subscribe [this opts]
    (locking global-monitor
      (when (:on-value opts)
        (println "WARN: provided 'on-value' callback with no watch key. Removing callback."
                 {:opts opts}))
      (subscribe* this nil (dissoc opts :on-value))))
  (subscribe [this]
    (locking global-monitor
      (subscribe* this nil nil)))
  (unsubscribe [this key]
    (locking global-monitor
      (unsubscribe* this key)))

  clojure.lang.IDeref
  (deref [this] (deref* this))

  Object
  (toString [this] (str @this)))

(defn make-reaction
  [computation-fn]
  (Reaction.
   computation-fn
   (atom :tidy/none)
   (atom :tidy/idle)
   (atom {})
   (atom #{})
   (atom 0)
   (LinkedBlockingQueue. 1024)))

(defmacro reaction
  [& body]
  `(make-reaction (fn [] ~@body)))

(defn reaction?
  [x]
  (satisfies? IReaction x))

(defn retain
  [reaction]
  (locking global-monitor
    (when (disposed? reaction)
      (throw
       (ex-info
        "Can not retain a disposed reaction"
        {:reaction reaction
         :id (System/identityHashCode reaction)})))
    (swap! (:ref-count reaction) inc)))

(defn release
  [reaction]
  (locking global-monitor
    (when (pos? @(:ref-count reaction))
      (when (zero? (swap! (:ref-count reaction) dec))
        (dispose* reaction)))))

(defn on-start
  ([reaction f] (on-start reaction (str "on_start_" (gensym)) f))
  ([reaction key f]
   (locking global-monitor
     (swap! (:listeners reaction) assoc key {:on-start f}))))

(defn on-dispose
  ([reaction f] (on-dispose reaction (str "on_dispose_" (gensym)) f))
  ([reaction key f]
   (locking global-monitor
     (swap! (:listeners reaction) assoc key {:on-dispose f}))))

(defn on-value
  ([reaction f] (on-value reaction (str "on_value_" (gensym)) f))
  ([reaction key f]
   (locking global-monitor
     (swap! (:listeners reaction) assoc key {:on-value f}))))

(defn remove-listener
  [reaction key]
  (locking global-monitor
    (swap! (:listeners reaction) dissoc key)))

;;; Record Print Helpers

(defmethod clojure.pprint/simple-dispatch Reaction [o]
  ((get-method clojure.pprint/simple-dispatch clojure.lang.IRecord) o))

(defmethod clojure.pprint/simple-dispatch RAtom [o]
  ((get-method clojure.pprint/simple-dispatch clojure.lang.IRecord) o))

(prefer-method print-method clojure.lang.IDeref clojure.lang.IPersistentMap)
(prefer-method print-method clojure.lang.IDeref clojure.lang.IRecord)
(prefer-method print-method clojure.lang.IDeref java.util.Map)

;;; Private

(defn- listeners
  [reaction type]
  (->> reaction
       :listeners
       deref
       vals
       (filter type)))

(defn value-listeners [reaction] (listeners reaction :on-value))
(defn start-listeners [reaction] (listeners reaction :on-start))
(defn dispose-listeners [reaction] (listeners reaction :on-dispose))

(defn- subscribe*
  [reaction key opts]
  (retain reaction)
  (let [{:keys [on-value on-dispose on-start]} opts]
    (when (or on-dispose on-start on-value)
      (swap! (:listeners reaction) assoc key opts))
    (run* reaction (select-keys opts run-opt-keys))))

(defn- unsubscribe*
  [reaction key]
  (swap! (:listeners reaction) dissoc key)
  (release reaction))

(defn- dispose*
  [reaction]
  (when-not (disposed? reaction)
    (reset! (:state reaction) :tidy/disposed)
    (doseq [c @(:inputs reaction)]
      (cond
        (reaction? c) (unsubscribe* c reaction)
        (instance? clojure.lang.IRef c) (remove-watch c reaction)
        :else (throw (ex-info "Unable to unsubscribe from unrecognized node type" {:node c}))))
    (doseq [{:keys [on-dispose]} (dispose-listeners reaction)]
      (on-dispose))))

(defn- downstream-reactions
  [reaction]
  (->> (:listeners reaction)
       deref
       (filter (fn [[k v]]
                 (and (:on-value v)
                      (reaction? k))))
       (map first)))

(defn upstream-reactions
  [reaction]
  (filter reaction? @(:inputs reaction)))

(defn upstream-stabilized?
  [reaction]
  (->> (upstream-reactions reaction)
       (map (comp deref :state))
       (every? (partial = :tidy/started))))

(defn downstream-stabilized?
  [reaction]
  (->> (downstream-reactions reaction)
       (map (comp deref :state))
       (every? (partial = :tidy/started))))

(defn stabilized?
  "A reaction is stabilized once it has been run at least once, and all
  upstream reactions are also stabilized. This allows us to determine whether
  the reaction graph is in a state where it can accept values, as well as to
  notify listeners when the graph has 'started' (i.e. stabilized)"
  [reaction]
  (and (= :tidy/started @(:state reaction))
       (upstream-stabilized? reaction)
       (downstream-stabilized? reaction)))

(defn- stabilize-graph!
  [reaction]
  (doseq [w (upstream-reactions reaction)]
    (stabilize-graph! w))
  (reset! (:state reaction) :tidy/started))

(defn- notify-started
  [reaction]
  (doseq [reaction (upstream-reactions reaction)] (notify-started reaction))
  (doseq [{:keys [on-start]} (start-listeners reaction)] (on-start)))

(defn- run*
  [reaction opts]
  (let [first-run? (= :tidy/idle @(:state reaction))]
    (when first-run? (reset! (:state reaction) :tidy/first-run))
    (let [captured (atom #{})
          {:keys [computation-fn]} reaction]
      (binding [*deref-context* {:reaction reaction
                                 :captured captured}]
        (let [result (computation-fn)
              inputs-old @(:inputs reaction)
              inputs-new @captured
              on-value (fn [c]
                         (fn [value & [ctx]]
                           (if-not (disposed? reaction)
                             ((if (and (not= ctx :ratom) *deref-context*)
                                run*
                                run)
                              reaction (select-keys opts run-opt-keys))
                             (throw
                              (ex-info
                               "Reaction is disposed"
                               {:reaction reaction})))))]
          (reset! (:inputs reaction) inputs-new)
          (reset! (:value reaction) result)
          (when-not (.offer (:value-queue reaction) {:result result})
            (prn* (str "Unable to report reaction result due to full queue" {:result result})))
          (doseq [c (difference inputs-new inputs-old)]
            (let [on-value (on-value c)]
              (cond

                (reaction? c)
                (subscribe*
                 c reaction
                 (merge
                  {:on-value on-value
                   :on-dispose (fn [] (swap! (:inputs reaction) disj c))}
                  (select-keys opts run-opt-keys)))

                (instance? clojure.lang.IRef c)
                (add-watch
                 c reaction
                 (fn [_ _ _ value]
                   (on-value value :ratom)))

                :else (throw (ex-info "Unable to subscribe to unrecognized node type" {:node c})))))
          (doseq [c (difference inputs-old inputs-new)]
            (cond
              (reaction? c) (unsubscribe* c reaction)
              (instance? clojure.lang.IRef c) (remove-watch c reaction)
              :else (throw (ex-info "Unable to unsubscribe from unrecognized node type" {:node c})))))))

    (when first-run? (reset! (:state reaction) :tidy/started))

    (when-let [{:keys [result]} (.poll (:value-queue reaction))]
      (when (and (initialized? reaction)
                 (upstream-stabilized? reaction))
        (doseq [{:keys [on-value]} (value-listeners reaction)]
          (on-value result))))

    (when (and first-run? (stabilized? reaction))
      (notify-started reaction))))

(defn- deref*
  [this]
  (cond

    *deref-context*
    (let [value @(:value this)]
      (when-let [captured (:captured *deref-context*)]
        (swap! captured conj this))
      (when (not= value :tidy/none) value))

    (:computation-fn this)
    ((:computation-fn this))

    :else
    (let [value @(:value this)]
      (when (not= value :tidy/none) value))))
