(ns tidy.core
  (:require [clojure.core.async :as core-async]
            [clojure.main :refer [repl-read]]
            [utilis.fn :refer [fsafe]]
            [utilis.map :refer [compact map-vals map-keys]]
            [clojure.set :refer [difference]]
            [clojure.string :as st]
            [tidy.core :as tidy])
  (:import [java.util.concurrent LinkedBlockingQueue]))

;;; Declarations

(declare run* dispose* subscribe* unsubscribe* deref* stabilized?)

(def ^:dynamic *deref-context* nil)
(def ^:private run-opt-keys [:silent? :best-effort?])

(def prn-monitor (Object.))
(defn prn*
  [& args]
  (locking prn-monitor
    (apply prn args)))

(defprotocol IReaction
  (dispose [this])
  (disposed? [this])
  (run [this] [this opts]))

(defprotocol ISubscribable
  (subscribe
    [this]
    [this opts]
    [this key opts])
  (unsubscribe [this key]))

(defprotocol IInitializable
  (initialized? [this]))

(defprotocol IReactiveAtom)

;;; Records

(defrecord RAtom [value]
  IReactiveAtom

  IInitializable
  (initialized? [this]
    (not= :tidy/none @value))

  clojure.lang.IDeref
  (deref [this] (deref* this))

  clojure.lang.IAtom
  (reset [this new-value] (.reset ^clojure.lang.IAtom value new-value))
  (swap [this f]          (.swap ^clojure.lang.IAtom value f))
  (swap [this f x]        (.swap ^clojure.lang.IAtom value f x))
  (swap [this f x y]      (.swap ^clojure.lang.IAtom value f x y))
  (swap [this f x y more] (.swap ^clojure.lang.IAtom value x y more))

  clojure.lang.IRef
  (addWatch [this key f]  (.addWatch ^clojure.lang.IAtom value key f) this)
  (removeWatch [this key] (.removeWatch ^clojure.lang.IAtom value key) this)
  (getWatches [this]      (.getWatches value))

  Object
  (toString [this] (str @this)))

(defn ratom?
  [x]
  (satisfies? IReactiveAtom x))

(defn ratom
  ([] (ratom :tidy/none))
  ([initial-state] (RAtom. (atom initial-state))))

(defrecord Reaction [computation-fn
                     value state
                     value-listeners
                     state-listeners
                     inputs
                     notify-queue]
  IReaction
  (dispose [this]   (dispose* this))
  (disposed? [this] (= :tidy/disposed @state))
  (run [this]       (run* this nil))
  (run [this opts]  (run* this opts))

  IInitializable
  (initialized? [this]
    (and (= :tidy/started @state)
         (not= :tidy/none @value)
         (every? initialized? @inputs)))

  ISubscribable
  (subscribe [this key opts] (subscribe* this key opts))
  (subscribe [this opts]
    (when (:on-value opts)
      (println "WARN: provided 'on-value' callback with no watch key. Removing callback."
               {:opts opts}))
    (subscribe* this nil (dissoc opts :on-value)))
  (subscribe [this]          (subscribe* this nil nil))
  (unsubscribe [this key]    (unsubscribe* this key))

  clojure.lang.IDeref
  (deref [this] (deref* this))

  Object
  (toString [this] (str @this)))

(defn make-reaction
  [computation-fn]
  (Reaction.
   computation-fn
   (atom :tidy/none)
   (atom :tidy/idle)
   (atom {})
   (atom [])
   (atom #{})
   (atom [])))

(defmacro reaction
  [& body]
  `(make-reaction (fn [] ~@body)))

(defn reaction?
  [x]
  (satisfies? IReaction x))

;;; Record Print Helpers

(defmethod clojure.pprint/simple-dispatch Reaction [o]
  ((get-method clojure.pprint/simple-dispatch clojure.lang.IRecord) o))

(defmethod clojure.pprint/simple-dispatch RAtom [o]
  ((get-method clojure.pprint/simple-dispatch clojure.lang.IRecord) o))

(prefer-method print-method clojure.lang.IDeref clojure.lang.IPersistentMap)
(prefer-method print-method clojure.lang.IDeref clojure.lang.IRecord)
(prefer-method print-method clojure.lang.IDeref java.util.Map)

;;; Private

(defn- subscribe*
  [reaction key opts]
  (locking reaction
    (let [{:keys [on-value on-dispose on-start]} opts
          stabilized? (stabilized? reaction)]
      (when (or on-dispose on-start)
        (->> [:on-dispose :on-start]
             (select-keys opts)
             (swap! (:state-listeners reaction) conj)))
      (when (fn? on-value)
        (swap!
         (:value-listeners reaction)
         assoc key
         {:on-value on-value}))
      (cond

        (and (not (:silent? opts))
             (= @(:state reaction) :tidy/idle))
        (run reaction (select-keys opts run-opt-keys))

        stabilized?
        (when-let [on-start (:on-start opts)]
          (on-start))

        :else nil))))

(defn- unsubscribe*
  [reaction key]
  (when (locking reaction
          (let [pre-listener-count (count @(:value-listeners reaction))]
            (not (seq (swap! (:value-listeners reaction) dissoc key)))))
    (dispose reaction)))

(defn- downstream-reactions
  [reaction]
  (->> @(:value-listeners reaction)
       keys
       (filter reaction?)))

(defn- downstream-started?
  [reaction]
  (->> (downstream-reactions reaction)
       (map (comp deref :state))
       (every? (partial = :tidy/started))))

(defn upstream-reactions
  [reaction]
  (filter reaction? @(:inputs reaction)))

(defn- upstream-started?
  [reaction]
  (->> (upstream-reactions reaction)
       (map (comp deref :state))
       (every? (partial = :tidy/started))))

(defn- notify-started
  [reaction]
  (doseq [reaction (upstream-reactions reaction)]
    (notify-started reaction))
  (doseq [{:keys [on-start]} (->> reaction
                                  :state-listeners
                                  deref
                                  (filter :on-start))]
    (on-start)))

(defn- notify-value
  [reaction value]
  (doseq [{:keys [on-value]} (->> reaction
                                  :value-listeners
                                  deref vals
                                  (filter :on-value))]
    (on-value value)))

(defn- notify-dispose
  [reaction]
  (doseq [{:keys [on-dispose]} (->> reaction
                                    :state-listeners
                                    deref
                                    (filter :on-dispose))]
    (on-dispose)))

(defn- dispose*
  [reaction]
  (doseq [input (locking reaction
                  (doseq [w (upstream-reactions reaction)]
                    (swap! (:value-listeners w) dissoc reaction))
                  (reset! (:state reaction) :tidy/disposed)
                  (notify-dispose reaction)
                  @(:inputs reaction))]

    (cond

      (reaction? input)
      (unsubscribe input reaction)

      (ratom? input)
      (remove-watch input reaction)

      (instance? clojure.lang.IRef reaction)
      (remove-watch input reaction)

      :else nil)))

(defn upstream-stabilized?
  [reaction]
  (->> (upstream-reactions reaction)
       (map (comp deref :state))
       (every? (partial = :tidy/started))))

(defn downstream-stabilized?
  [reaction]
  (->> (downstream-reactions reaction)
       (map (comp deref :state))
       (every? (partial = :tidy/started))))

(defn stabilized?
  "A reaction is stabilized once it has been run at least once, and all
  upstream reactions are also stabilized. This allows us to determine whether
  the reaction graph is in a state where it can accept values, as well as to
  notify listeners when the graph has 'started' (i.e. stabilized)"
  [reaction]
  (and (= :tidy/started @(:state reaction))
       (upstream-stabilized? reaction)
       (downstream-stabilized? reaction)))

(defn- stabilize-graph!
  [reaction]
  (doseq [w (upstream-reactions reaction)]
    (stabilize-graph! w))
  (reset! (:state reaction) :tidy/started))

(defn- run*
  [reaction opts]
  (let [{:keys [to-unsub]}
        (locking reaction
          (let [first-run? (when (= :tidy/idle @(:state reaction))
                             (reset! (:state reaction) :tidy/first-run)
                             true)
                captured (atom #{})
                {:keys [computation-fn]} reaction]
            (binding [*deref-context*
                      {:reaction reaction
                       :captured captured}]
              (let [result (computation-fn)
                    inputs-old @(:inputs reaction)
                    inputs-new @captured
                    on-value (fn [c]
                               (fn [value]
                                 (if-not (disposed? reaction)
                                   (run reaction
                                     (select-keys opts run-opt-keys))
                                   (throw
                                    (ex-info
                                     "Reaction is disposed"
                                     {:reaction reaction})))))
                    notify? (fn []
                              (and (stabilized? reaction)
                                   (or (:best-effort? opts)
                                       (initialized? reaction))))]

                (reset! (:inputs reaction) inputs-new)
                (reset! (:value reaction) result)
                (when (notify?) (swap! (:notify-queue reaction) #(conj (vec %) result)))

                (doseq [c (difference inputs-new inputs-old)]
                  (let [on-value (on-value c)]
                    (cond

                      (reaction? c)
                      (subscribe
                       c reaction
                       (merge
                        {:on-value on-value
                         :on-dispose (fn [] (swap! (:inputs reaction) disj c))}
                        (select-keys opts run-opt-keys)))

                      (instance? clojure.lang.IRef c)
                      (add-watch
                       c reaction
                       (fn [_ _ _ value]
                         (on-value value)))

                      :else (throw (ex-info "Unable to subscribe to unrecognized node type" {:node c})))))

                (when (and first-run?
                           (downstream-stabilized? reaction))
                  (stabilize-graph! reaction))

                (when (and (seq @(:notify-queue reaction)) (notify?))
                  (notify-value
                   reaction
                   (let [v (first @(:notify-queue reaction))]
                     (swap! (:notify-queue reaction) rest)
                     v)))

                (when (and first-run? (stabilized? reaction))
                  (notify-started reaction))

                {:to-unsub (difference inputs-old inputs-new)}))))]
    (doseq [c to-unsub]
      (cond
        (reaction? c) (unsubscribe c reaction)
        (instance? clojure.lang.IRef c) (remove-watch c reaction)
        :else (throw (ex-info "Unable to unsubscribe from unrecognized node type" {:node c}))))))

(defn- deref*
  [this]
  (cond

    *deref-context*
    (let [value @(:value this)]
      (when-let [captured (:captured *deref-context*)]
        (swap! captured conj this))
      (when (not= value :tidy/none) value))

    (:computation-fn this)
    ((:computation-fn this))

    :else
    (let [value @(:value this)]
      (when (not= value :tidy/none) value))))
