(ns re-frame-auth.flows.instagram
  (:require [re-frame-auth.protocols :as proto]
            [re-frame-auth.flows.core :as flow]
            [re-frame-auth.stores.core :as store]
            [utilis.map :refer [compact]]
            [org.httpkit.client :as http]
            [clojure.data.json :as json]
            [clojure.string :as st]
            [clojure.spec.alpha :as s]))

;;; Declarations

(declare authenticate*)

(s/def :instagram-flow/token (s/and string? seq))
(s/def ::instagram-flow
  (s/keys :req-un [:instagram-flow/token]))

;;; API

(defrecord InstagramFlow []
  proto/AuthenticationFlow
  (authenticate [_ request] (authenticate* request)))

(defn instagram-flow
  []
  (flow/authenticator (InstagramFlow.)))

;;; Private

(defn instagram-token->user
  [{:keys [token]}]
  (let [response @(http/get
                   "https://api.instagram.com/v1/users/self/"
                   {:query-params {"access_token" token}})]
    (when (= (:status response) 200)
      (let [{:keys [id profile_picture full_name bio]} (-> response :body (json/read-json true) :data)
            [first-name last-name] (st/split (str full_name) #" ")]
        (compact
         {:instagram-id id
          :first-name first-name
          :last-name last-name
          :bio bio
          :display-name (->> [first-name last-name]
                             (remove nil?)
                             (map (comp st/capitalize st/lower-case))
                             (st/join " "))
          :profile-image-url profile_picture})))))

(defn- authenticate*
  [{:keys [params] :as request}]
  (try
    (cond
      (not (s/valid? ::instagram-flow params))
      (flow/not-authenticated
       {:type :flow/validation-error
        :cause :instagram-flow/invalid-intagram-token
        :message (s/explain-str ::instagram-flow params)})
      :else
      (or (let [auth-store (:auth/auth-store request)]
            (when-let [instagram-user (instagram-token->user {:token (:token params)})]
              (let [auth-record (store/auth-record
                                 :instagram
                                 {:instagram-id
                                  (:instagram-id instagram-user)}
                                 :primary-key :instagram-id)]
                (if-let [match (proto/find-auth auth-store auth-record)]
                  (flow/authenticated
                   {:new-user? false
                    :user {:id (:user-id match)}})
                  (flow/new-user auth-record {:instagram instagram-user})))))
          (flow/not-authenticated
           {:type :flow/validation-error
            :cause :instagram-flow/user-not-found
            :message "Instagram user not found for token"})))

    (catch Exception e
      (flow/not-authenticated
       {:throwable e
        :type :instagram-flow/error
        :cause :instagram-flow/exception}))))
