(ns re-frame-auth.fx
  (:require [re-frame-auth.fx.http]
            [re-frame-auth.fx.storage]
            [re-frame.core :refer [dispatch subscribe reg-event-fx reg-sub]]
            [utilis.types.string :refer [->string]]
            [utilis.map :refer [compact]]
            [clojure.string :as st]
            [cljs.pprint :refer [pprint]]))

;;; Declarations

(enable-console-print!)

(def ^:private require? (exists? js/require))
(def ReactNative (when require? (js/require "react-native")))

(def ^:private access-token-storage-key "auth/access-token")
(def ^:private id-token-storage-key "auth/id-token")
(def ^:private refresh-token-storage-key "auth/refresh-token")

(def ^:private react-native? (boolean ReactNative))
(def ^:private web? (not react-native?))
(def ^:private cookie-auth? web?)

(declare valid-tokens? concat-event prep-token-params)

;;; Subscriptions

(reg-sub
 :auth/access-token
 (fn [db] (::access-token db)))

(reg-sub
 :auth/id-token
 (fn [db] (::id-token db)))

(reg-sub
 :auth/refresh-token
 (fn [db] (::refresh-token db)))

;;; API

(reg-event-fx
 :auth/login
 (fn [{:keys [db]} [_ {:keys [credentials url on-success on-failure] :as params}]]
   {:dispatch [:auth/post params]}))

(reg-event-fx
 :auth/sign-up
 (fn [{:keys [db]} [_ {:keys [url on-success on-failure] :as params}]]
   {:dispatch [:auth/post params]}))

(reg-event-fx
 :auth/login-or-sign-up
 (fn [{:keys [db]} [_ {:keys [url on-success on-failure] :as params}]]
   {:dispatch [:auth/post params]}))

(reg-event-fx
 :auth/token-login
 (fn [{:keys [db]} [_ {:keys [url on-success on-failure] :as params}]]
   {:dispatch [:auth/post-with-tokens (assoc params :refresh-on-failure? true)]}))

(reg-event-fx
 :auth/logout
 (fn [{:keys [db]} [_ {:keys [url on-success on-failure] :as params}]]
   (when-let [events
              (->> [(when react-native? [:auth/clear-tokens])
                    (when (seq url) [:auth/post-with-tokens params])]
                   (remove nil?)
                   not-empty)]
     {:dispatch-n events})))

(reg-event-fx
 :auth/refresh-access-token
 (fn [{:keys [db]} [_ {:keys [url on-success on-failure] :as params}]]
   {:dispatch [:auth/post-with-tokens params]}))

;;; Token Helpers

(reg-event-fx
 :auth/post-with-load-tokens-result
 (fn [{:keys [db]} [_ params tokens]]
   {:dispatch
    [:auth/post
     (assoc params :credentials
            {:tokens (prep-token-params tokens)})]}))

(reg-event-fx
 :auth/post-with-tokens
 (fn [{:keys [db]} [_ {:keys [on-failure] :as params}]]
   {:dispatch
    (if cookie-auth?
      [:auth/post params]
      [:auth/load-tokens
       {:on-success [:auth/post-with-load-tokens-result params]
        :on-failure on-failure}])}))

;;; Global Authentication Event Handlers

(reg-event-fx
 :auth/authentication-result
 (fn [{:keys [db]} [_ {:keys [on-success on-failure url refresh-url refresh-on-failure?] :as params} result]]
   (if (:success? result)
     (if cookie-auth? ;; If using cookie, assume everything is fine if server says so
       {:dispatch (concat-event on-success)}
       (if-let [tokens (-> result :body (js->clj :keywordize-keys true) :tokens)]
         {:dispatch-n
          (cons
           [:auth/store-tokens {:tokens tokens}]
           [(concat-event on-success tokens)])}
         {:dispatch (concat-event on-failure {:reason :no-tokens-in-response :response result})}))
     {:dispatch
      (if (and refresh-on-failure? refresh-url
               (or cookie-auth? (-> params :credentials :tokens seq)))
        [:auth/refresh-access-token
         (assoc params
                :url refresh-url
                :refresh-on-failure? false)]
        (concat-event
         on-failure
         {:reason :authentication-failed
          :response result}))})))

(reg-event-fx
 :auth/post
 (fn [{:keys [db]} [_ {:keys [credentials url on-success on-failure] :as params}]]
   {:auth-fx/http
    {:method :post
     :url url
     :params (compact
              (merge
               credentials
               {:cookie cookie-auth?}))
     :on-success [:auth/authentication-result params]
     :on-failure [:auth/authentication-result params]}}))

;;; Token Storage

(reg-event-fx
 :auth/clear-tokens
 (fn [{:keys [db]}]
   {:storage/delete
    {:k [access-token-storage-key
         id-token-storage-key
         refresh-token-storage-key]}
    :db (dissoc db
                ::access-token
                ::id-token
                ::refresh-token)}))

(reg-event-fx
 :auth/load-tokens
 (fn [{:keys [db]} [_ {:keys [on-success on-failure]}]]
   {:storage/get
    {:k [access-token-storage-key
         id-token-storage-key
         refresh-token-storage-key]
     :on-success on-success
     :on-failure on-failure}}))

(reg-event-fx
 :auth/store-tokens
 (fn [{:keys [db]} [_ {:keys [tokens on-success on-failure]}]]
   {:db (merge db (compact
                   {::access-token (:access-token tokens)
                    ::id-token (:id-token tokens)
                    ::refresh-token (:refresh-token tokens)}))
    :storage/put
    {:values (compact
              {access-token-storage-key (:access-token tokens)
               id-token-storage-key (:id-token tokens)
               refresh-token-storage-key (:refresh-token tokens)})
     :on-success on-success
     :on-failure on-failure}}))

(reg-event-fx
 :auth/no-op
 (fn [{:keys [db]} [_ ]]
   ))

;;; Private

(defn- valid-token?
  [token]
  (and (string? token) (seq (st/trim token))))

(defn- valid-tokens?
  [tokens]
  (every? valid-token? (vals tokens)))

(defn- concat-event
  [event & args]
  (or (when (vector? event)
        (vec (concat event args)))
      [:auth/no-op]))

(defn- ensure-string
  [x]
  (when (string? x) x))

(defn- prep-token-params
  [tokens]
  (when-let [tokens (compact
                     {:access-token (ensure-string (get tokens access-token-storage-key))
                      :id-token (ensure-string (get tokens id-token-storage-key))
                      :refresh-token (ensure-string (get tokens refresh-token-storage-key))})]
    (js/JSON.stringify (clj->js tokens))))
