(ns re-frame-auth.stores.memory
  (:require [re-frame-auth.protocols :as proto]
            [re-frame-auth.util.id :refer [generate-auth-key uuid]]
            [utilis.types.keyword :refer [->keyword]]
            [integrant.core :as ig]))

;;; Records

(defrecord UserMemoryStore [backing-atom]
  proto/UserStore
  (find-user [_ user-id]
    (get @backing-atom user-id))
  (create-user! [this user]
    (let [user (update user :id #(or % (uuid)))]
      (when (proto/find-user this (:id user))
        (throw (ex-info "User already exists" {:user user})))
      (get (swap! backing-atom assoc (:id user) user) (:id user))))
  (delete-user! [this user-id]
    (swap! backing-atom dissoc user-id)))

(defrecord AuthMemoryStore [backing-atom]
  proto/AuthStore
  (find-auth [_ auth]
    (->> @backing-atom vals
         (filter #(= (get auth (:primary-key auth))
                     (get % (:primary-key auth))))
         first))
  (create-auth! [this auth]
    (let [auth (assoc auth :id (generate-auth-key auth))
          result (swap! backing-atom
                        #(if (proto/find-auth this auth)
                           %
                           (assoc % (:id auth) auth)))]
      (when-not (get result (:id auth))
        (throw (ex-info "Auth already exists" {:auth auth})))))
  (delete-auth! [this auth]
    (swap! backing-atom dissoc (generate-auth-key auth))
    nil)
  (create-refresh-token! [_ claims]
    (swap! backing-atom assoc-in [:refresh-tokens (:secret claims)] claims)
    nil)
  (revoke-refresh-token! [_ claims]
    (swap! backing-atom assoc-in [:refresh-tokens (:secret claims) :revoked] true)
    nil)
  (refresh-token-revoked? [_ claims]
    (let [result (get-in @backing-atom [:refresh-tokens (:secret claims)])]
      (or (nil? result)
          (-> result :revoked boolean)))))

;;; Integrant

(defmethod ig/init-key :user-store/memory
  [_ {:keys []}]
  (map->UserMemoryStore {:backing-atom (atom {})}))

(defmethod ig/halt-key! :user-store/memory
  [_ {:keys []}]
  )

(defmethod ig/init-key :auth-store/memory
  [_ {:keys []}]
  (map->AuthMemoryStore {:backing-atom (atom {})}))

(defmethod ig/halt-key! :auth-store/memory
  [_ {:keys []}]
  )
