(ns re-frame-auth.fx.storage
  (:refer-clojure :exclude [get])
  (:require [re-frame.core :refer [reg-event-fx reg-fx reg-sub dispatch]]
            [utilis.types.string :refer [->string]]
            [utilis.types.keyword :refer [->keyword]]
            [utilis.map :refer [compact map-keys]]
            [clojure.string :as st]
            [alandipert.storage-atom :refer [local-storage]]))

;;; Declarations

(enable-console-print!)

(def ^:private require? (exists? js/require))

(def ReactNative (when require? (js/require "react-native")))
(def AsyncStorage (when ReactNative (.-AsyncStorage ReactNative)))
(def react-native? (boolean AsyncStorage))
(def web-token-store
  (when-not react-native?
    (local-storage (atom {}) :tokens)))

(declare safe-dispatch ->ks)

;;; Implementation

(defn put!
  [{:keys [values on-success on-failure]}]
  (if react-native?
    (-> AsyncStorage
        (.multiSet (clj->js (seq values)))
        (.then (partial safe-dispatch on-success))
        (.catch (partial safe-dispatch on-failure)))
    (swap! web-token-store merge values)))

(defn delete!
  [{:keys [k on-success on-failure]}]
  (if react-native?
    (-> AsyncStorage
        (.multiRemove (clj->js (->ks k)))
        (.then (partial safe-dispatch on-success))
        (.catch (partial safe-dispatch on-failure)))
    (swap! web-token-store
           (fn [tokens]
             (apply dissoc tokens (->ks k))))))

(defn get
  [{:keys [k on-success on-failure]}]
  (if react-native?
    (-> AsyncStorage
        (.multiGet (clj->js (->ks k)))
        (.then (fn [kv-pairs]
                 (let [kv-pairs (into {} (js->clj kv-pairs :keywordize-keys true))]
                   (safe-dispatch on-success kv-pairs))))
        (.catch (partial safe-dispatch on-failure)))
    (safe-dispatch on-success (select-keys @web-token-store (->ks k)))))

;;; API

(reg-fx :storage/put put!)
(reg-fx :storage/get get)
(reg-fx :storage/delete delete!)

;;; Private

(defn- safe-dispatch
  [event & args]
  (when (and event (vector? event))
    (dispatch (vec (concat event args)))))

(defn- ->ks
  [k-or-ks]
  (if (and (coll? k-or-ks)
           (sequential? k-or-ks))
    k-or-ks
    [k-or-ks]))
