(ns re-frame-auth.flows.firebase
  (:require [re-frame-auth.protocols :as proto]
            [re-frame-auth.flows.core :as flow]
            [re-frame-auth.stores.core :as store]
            [utilis.map :refer [compact]]
            [org.httpkit.client :as http]
            [clojure.data.json :as json]
            [clojure.string :as st]
            [clojure.spec.alpha :as s])
  (:import [com.google.firebase.auth FirebaseAuth]
           [com.google.firebase FirebaseOptions FirebaseOptions$Builder FirebaseApp]
           [com.google.firebase.messaging Message Notification FirebaseMessaging]
           [java.io FileInputStream]
           [com.google.auth.oauth2 GoogleCredentials]))

;;; Declarations

(declare authenticate*)

(s/def :firebase-flow/id-token (s/and string? seq))
(s/def ::firebase-flow
  (s/keys :req-un [:firebase-flow/id-token]))

;;; API

(defrecord FirebaseFlow [firebase-instance]
  proto/AuthenticationFlow
  (authenticate [_ request] (authenticate* firebase-instance request)))

(defn firebase-flow
  [firebase-instance]
  (flow/authenticator (FirebaseFlow. firebase-instance)))

(defn firebase-instance
  [{:keys [app-name service-account-key-path]
    :or {app-name "ClojureFirebaseApp"}}]
  (-> (FirebaseOptions$Builder.)
      (.setCredentials
       (GoogleCredentials/fromStream
        (FileInputStream. service-account-key-path)))
      (.build)
      (FirebaseApp/initializeApp app-name)))

;;; Private

(defn- id-token->user
  [firebase-instance id-token]
  (try (when-let [auth-instance (FirebaseAuth/getInstance firebase-instance)]
         (when-let [decoded (.verifyIdToken auth-instance id-token)]
           (let [uid (.getUid decoded)
                 user (.getUser auth-instance uid)]
             (compact
              {:firebase-id uid
               :display-name (.getDisplayName user)
               :profile-image-url (when-let [image-url (.getPhotoUrl user)]
                                    (if (re-find #"https://graph.facebook.com" image-url)
                                      (str image-url "?type=large")
                                      image-url))}))))
       (catch Exception e
         (prn e)
         nil)))

(defn- authenticate*
  [firebase-instance {:keys [params] :as request}]
  (try
    (cond
      (not (s/valid? ::firebase-flow params))
      (flow/not-authenticated
       {:type :flow/validation-error
        :cause :firebase-flow/invalid-id-token
        :message (s/explain-str ::firebase-flow params)})
      :else
      (or (let [auth-store (:auth/auth-store request)]
            (when-let [firebase-user (id-token->user firebase-instance (:id-token params))]
              (let [auth-record (store/auth-record
                                 :firebase firebase-user
                                 :primary-key :firebase-id)]
                (if-let [match (proto/find-auth auth-store auth-record)]
                  (flow/authenticated
                   {:new-user? false
                    :user {:id (:user-id match)}})
                  (flow/new-user auth-record {:firebase firebase-user})))))
          (flow/not-authenticated
           {:type :flow/validation-error
            :cause :firebase-flow/user-not-found
            :message "Firebase user not found for token"})))

    (catch Exception e
      (flow/not-authenticated
       {:throwable e
        :type :firebase-flow/error
        :cause :firebase-flow/exception}))))
