(ns re-frame-auth.flows.facebook
  (:require [re-frame-auth.flows.core :as flows]
            [re-frame-auth.stores.core :as stores]
            [utilis.map :refer [compact]]
            [utilis.types.keyword :refer [->keyword]]
            [org.httpkit.client :as http]
            [clojure.data.json :as json]
            [clojure.string :as st]))

;;; Declarations

(declare facebook-token->user)

;;; API

#_(defmethod flows/authenticate :facebook
    [{:keys [params] :as request}]
    (let [auth-store (:re-frame-auth.core/auth-store request)
          facebook-app-id (:re-frame-auth.core/facebook-app-id request)
          facebook-app-secret (:re-frame-auth.core/facebook-app-secret request)]
      (when (and facebook-app-id facebook-app-secret)
        (when-let [facebook-user (facebook-token->user
                                  {:token (:token params)
                                   :app-id facebook-app-id
                                   :app-secret facebook-app-secret})]
          (let [auth-record {:auth-type :facebook
                             :primary-key :facebook-id
                             :facebook-id (:facebook-id facebook-user)}]
            (if-let [auth (stores/find-auth
                           {:store auth-store
                            :auth auth-record})]
              {:auth auth :exists? true}
              {:auth auth-record
               :user (dissoc facebook-user :facebook-id)
               :exists? false}))))))

;;; Private

(defn- verify-facebook-token
  [{:keys [app-id app-secret token]}]
  (-> @(http/get
        "https://graph.facebook.com/v2.6/debug_token"
        {:query-params {"input_token" token
                        "access_token" (str app-id "|" app-secret)}})
      :status (= 200)))

(defn- facebook-token->user
  [{:keys [token app-id app-secret]}]
  (when (verify-facebook-token
         {:app-id app-id
          :app-secret app-secret
          :token token})
    (let [response @(http/get
                     "https://graph.facebook.com/v2.6/me"
                     {:query-params
                      {"access_token" token
                       "fields" (st/join "," ["id"
                                              "name"
                                              "first_name"
                                              "last_name"
                                              "middle_name"
                                              "gender"
                                              "link"
                                              "email"
                                              "picture"
                                              "birthday"])}})]
      (when (= (:status response) 200)
        (let [{:keys [id name first_name middle_name last_name
                      gender link picture birthday]} (-> response
                                                         :body
                                                         (json/read-json true))]
          (compact
           {:facebook-id id
            :birthday birthday
            :first-name first_name
            :middle-name middle_name
            :last-name last_name
            :display-name (->> [first_name middle_name last_name]
                               (remove nil?)
                               (map (comp st/capitalize st/lower-case))
                               (st/join " "))
            :gender (when gender (->keyword gender))
            :profile-image-url (format "https://graph.facebook.com/v2.6/%s/picture?type=large" id)}))))))
