(ns re-frame-auth.flows.core
  (:require [re-frame-auth.protocols :as proto]
            [re-frame-auth.util.spec :refer [validate]]
            [re-frame-auth.stores.core]
            [utilis.map :refer [compact]]
            [clojure.spec.alpha :as s]))

;;; Records

(defrecord Authenticator [flow]
  proto/AuthenticationFlow
  (authenticate [_ request]
    (->> request
         (proto/authenticate flow)
         (validate :auth-flow/result))))

(defrecord Refresher [flow]
  proto/RefreshFlow
  (refresh [_ request]
    (->> request
         (proto/refresh flow)
         (validate :auth-flow/result))))

;;; API

(defn authenticator
  [flow]
  {:pre [(satisfies? proto/AuthenticationFlow flow)]}
  (map->Authenticator {:flow flow}))

(defn refresher
  [flow]
  {:pre [(satisfies? proto/RefreshFlow flow)]}
  (map->Refresher {:flow flow}))

(defn authentication-result
  [result]
  (validate :auth-flow/result result))

(defn authenticated
  ([] (authenticated nil))
  ([args]
   (authentication-result
    (assoc args :authenticated? true))))

(defn not-authenticated
  ([error] (not-authenticated error nil))
  ([error args]
   (authentication-result
    (assoc args
           :authenticated? false
           :error error))))

(defn new-user
  ([auth-record] (new-user auth-record nil))
  ([auth-record user]
   (authentication-result
    (compact
     {:new-user? true
      :auth-record auth-record
      :user user}))))

;;; Spec

(s/def :auth-flow-result-user/id string?)
(s/def :auth-flow-result/user
  (s/keys
   :opt-un [:auth-flow-result-user/id]))

(s/def :auth-flow-result-error/cause keyword?)
(s/def :auth-flow-result-error/type keyword?)
(s/def :auth-flow-result-error/message string?)
(s/def :auth-flow-result-error/throwable (partial instance? Throwable))
(s/def :auth-flow-result/error
  (s/keys
   :req-un [:auth-flow-result-error/type
            :auth-flow-result-error/cause]
   :opt-un [:auth-flow-result-error/message
            :auth-flow-result-error/throwable]))

(s/def :auth-flow-result/auth-record :auth-store/record)

(s/def :auth-flow-result-event/type keyword?)
(s/def :auth-flow-result-event/data any?)
(s/def :auth-flow-result-event/request any?)
(s/def :auth-flow-result-event/error any?)
(s/def :auth-flow-result/event
  (s/keys :req-un
          [:auth-flow-result-event/type]
          :opt-un
          [:auth-flow-result-event/request
           :auth-flow-result-event/error
           :auth-flow-result-event/data]))

(s/def :auth-flow-result/events
  (s/coll-of :auth-flow-result/event))

(s/def :auth-flow-result/new-user? boolean?)
(s/def :auth-flow-result/authenticated? boolean?)
(s/def :auth-flow/result
  (s/keys :opt-un
          [:auth-flow-result/authenticated?
           :auth-flow-result/new-user?
           :auth-flow-result/error
           :auth-flow-result/user
           :auth-flow-result/auth-record
           :auth-flow-result/events]))

(s/def :auth-flow/authenticator
  (partial instance? Authenticator))

(s/def :auth-flow/refresher
  (partial instance? Refresher))
