(ns re-frame-auth.core
  (:require [re-frame-auth.protocols :as proto]
            [re-frame-auth.flows.token :as token-flow]
            [re-frame-auth.stores.core]
            [re-frame-auth.flows.core]

            [clojure.data.json :as json]
            [clojure.spec.alpha :as s]
            [spec-tools.core :as sc]
            [re-frame-auth.util.spec :refer [validate]]

            [clj-time.coerce :as tc]
            [clj-time.core :as t]

            [re-frame-auth.util.id :as id :refer [uuid]]

            [utilis.types.keyword :refer [->keyword]]
            [utilis.types.string :refer [->string]]
            [utilis.map :refer [compact map-keys map-vals]]
            [utilis.fn :refer [fsafe]]

            [taoensso.timbre :as log]
            [clojure.string :as st]
            [clj-uuid :as uuid])
  (:import [org.joda.time DateTime]))

;;; Declarations

(declare tokens-response
         maximum-cookie-security?
         throwable?
         dispatch-events
         default-event-handler
         success-status-code)

;;; API

(defn extract-identity
  [id-token]
  (when (and (map? id-token)
             (seq id-token)
             (= (:token-use id-token) :id))
    (-> id-token
        (dissoc :secret :sub :exp :token-use)
        (assoc :user-id (:sub id-token)))))

(defn default-unauthorized-response
  ([] (default-unauthorized-response nil))
  ([request]
   (if (seq request)
     (assoc (-> request
                (assoc-in [:auth/cookie-options :expires] "Thu, 01 Jan 1970 00:00:01 GMT")
                (tokens-response
                 (map-vals
                  (constantly nil)
                  (:auth/token-generators request)))
                (dissoc :body :headers)) :status 401)
     {:status 401})))

(defn wrap-authentication
  "At a minimum, an 'auth-store' and 'user-store' must be provided, and an
  :access-token and :id-token generator must each be provided."
  [handler & {:keys [token-generators
                     auth-store
                     user-store
                     on-event
                     cookie-options]
              :or {on-event default-event-handler}
              :as args}]
  {:pre [(validate :auth/token-generators token-generators)
         (or (nil? cookie-options)
             (validate :auth/cookie-options cookie-options))
         (validate :store/auth auth-store)
         (validate :store/user user-store)]}

  ;; Warn maximum cookie security settings
  (when-not (maximum-cookie-security? cookie-options)
    (log/warn "Cookie security settings should be improved, e.g:\n"
              (with-out-str
                (clojure.pprint/pprint
                 {:path "/"
                  :http-only true
                  :same-site :strict
                  :domain "your-domain.com"
                  :secure true}))))

  (fn [request]
    (let [request (assoc request :auth/token-generators token-generators)
          tokens (token-flow/tokens request)]
      (-> request
          (assoc :auth/tokens tokens
                 :auth/access-token-expiry
                 (:exp (proto/parse
                        (:access-token token-generators)
                        (:access-token tokens)))
                 :auth/identity
                 (extract-identity
                  (proto/parse
                   (:id-token token-generators)
                   (:id-token tokens)))
                 :auth/on-event on-event
                 :auth/cookie-options (compact cookie-options)
                 :auth/user-store user-store
                 :auth/auth-store auth-store)
          handler))))

(defn request-refresh-tokens
  [request]
  (let [tokens (-> request
                   (update :auth/token-generators
                           #(when (map? %)
                              (select-keys % [:refresh-token])))
                   token-flow/parse-tokens)]
    (when (map? tokens)
      (filter
       (comp (partial = :refresh) :token-use)
       (vals tokens)))))

(defn revoke-refresh-tokens!
  [request]
  (doseq [claims (request-refresh-tokens request)]
    (proto/revoke-refresh-token!
     (:auth/auth-store request)
     claims)))

(defn authentication-result->action
  [authentication-result]
  (if (true? (:new-user? authentication-result))
    :auth/create
    :auth/login))

(defmulti process-authentication-result
  (fn [{:keys [request authentication-result actions]}]
    (let [authentication-result (validate :auth-flow/result authentication-result)
          action (authentication-result->action authentication-result)]

      (cond

        ;; Not a security feature, but this gives the client the flexibility of
        ;; creating login/sign-up only endpoints easily without having to
        ;; somehow provide hooks into the decision making graph.
        (not (get actions action))
        :auth/action-denied

        ;; An authentication error means the flow errored out before trying to
        ;; perform the actual authentication step (e.g. schema was bad)
        (or (:error authentication-result)
            (and (false? (:new-user? authentication-result))
                 (false? (:authenticated? authentication-result))))
        :auth/handle-authentication-error

        ;; Authentication succeeded!
        (and (not (:new-user? authentication-result))
             (true? (:authenticated? authentication-result)))
        :auth/handle-authentication-success

        ;; A new user is found
        (true? (:new-user? authentication-result))
        :auth/handle-new-user

        :else :default))))

(defn auth-handler
  "Given a 'request' and an 'authenticator', attempt to authenticate. The result
  of the authentication is an :auth-flow/result as defined in
  re-frame-auth.flows.core under the spec section.

  Either an unauthorized response or a response with auth tokens is returned."
  [request authenticator
   & [{:keys [actions
              tx-new-user
              allow-new-user?
              allow-login?
              unauthorized-response]
       :or {actions #{:auth/create
                      :auth/login}
            tx-new-user identity
            allow-new-user? (constantly true)
            allow-login? (constantly true)
            unauthorized-response default-unauthorized-response}}]]
  {:pre [(validate :auth-flow/authenticator authenticator)
         (validate :auth/actions-set actions)]}
  (let [request (update request :auth/on-event fsafe)
        result (proto/authenticate authenticator request)
        response (process-authentication-result
                  {:request request
                   :authentication-result result
                   :actions actions
                   :tx-new-user tx-new-user
                   :allow-new-user? allow-new-user?
                   :allow-login? allow-login?
                   :unauthorized-response unauthorized-response})]
    (dispatch-events
     (concat (:events result)
             (:events response))
     (:auth/on-event request))
    (:response response)))

(defn refresh-handler
  "This endpoint is used strictly to generate new access tokens given a valid
  or expired access token and a valid refresh token. When accepted, a new access
  token is delivered alongside the same refresh token, with its expiry bumped
  up."
  [request refresher
   & [{:keys [unauthorized-response allow-refresh?]
       :or {unauthorized-response default-unauthorized-response
            allow-refresh? (constantly true)}}]]
  {:pre [(validate :auth-flow/refresher refresher)]}
  (let [request (update request :auth/on-event fsafe)
        result (proto/refresh refresher request)
        response (process-authentication-result
                  {:request request
                   :authentication-result result
                   :actions #{:auth/login}
                   :unauthorized-response unauthorized-response
                   :allow-refresh? allow-refresh?})]
    (dispatch-events
     (concat (:events result)
             (:events response))
     (:auth/on-event request))
    (:response response)))

(defn logout-handler
  "When called, unset all authentication cookies if they exist, and expire the
  refresh token if present. Return no tokens in response."
  [request]
  (let [request (update request :auth/on-event fsafe)]
    (revoke-refresh-tokens! request)
    (dispatch-events
     [{:request request
       :type :auth/logout}]
     (:auth/on-event request))
    (default-unauthorized-response request)))

(defn sign-up!
  ([request authentication-result] (sign-up! request authentication-result nil))
  ([request {:keys [user auth-record] :as authentication-result} {:keys [attempts] :or {attempts 3}}]
   (let [auth-store (:auth/auth-store request)
         user-store (:auth/user-store request)]
     (if (seq auth-record)
       (let [timestamp (tc/to-long (t/now))
             generated-user-id? (boolean (:id user))
             user-id (or (:id user) (uuid))
             user (assoc user :created timestamp :id user-id)
             auth-record (assoc auth-record
                                :created timestamp
                                :user-id user-id)]
         (proto/create-auth! auth-store auth-record)
         (let [result (try (do (proto/create-user! user-store user) user)
                           (catch Exception e
                             (if generated-user-id? ::retry e)))]
           (cond

             (= result ::retry)
             (if (pos? attempts)
               (do (proto/delete-auth! auth-store auth-record)
                   (sign-up! request authentication-result {:attempts (dec attempts)}))
               (throw
                (ex-info
                 "User with that id already exists."
                 {:user user :auth-record auth-record})))

             (throwable? result) (throw result)

             :else result)))
       (throw
        (ex-info
         "Must provide an auth-record."
         {:authentication-result authentication-result}))))))

;;; Private

(defn- ->boolean
  [x]
  (boolean
   (cond

     (string? x)
     (= x "true")

     (integer? x)
     (= x 1)

     :else x)))

(defn- throwable?
  [x]
  (instance? Throwable x))

(defn- success-status-code
  [request]
  ({:get 200 :post 201} (:request-method request)))

(defn- cookie
  [value {:keys [path http-only same-site domain secure expires]
          :or {http-only true
               secure false}}]
  (let [cookie-options {:path path
                        :http-only http-only
                        :same-site same-site
                        :domain domain
                        :secure secure
                        :expires expires}]
    (merge {:value (str value)} (compact cookie-options))))

(defn- token-cookies
  [request tokens]
  (let [options (:auth/cookie-options request)]
    (->> tokens
         (map (fn [[k value]]
                [(->keyword k)
                 (cookie value
                         (-> options
                             (assoc :expires
                                    (-> (:auth/token-generators request)
                                        (get k)
                                        proto/expiry-time-ms
                                        (+ (tc/to-long (t/now)))
                                        tc/from-long))
                             compact))]))
         (into {}))))

(defn- tokens-response
  [request tokens]
  {:pre [(seq tokens)]}
  (if (-> request :params :cookie ->boolean)
    {:cookies (token-cookies request tokens)}
    {:body (json/write-str {:tokens tokens})
     :headers {"Content-Type" "application/json"}}))

(defn success-tokens-response
  [request authentication-result user-id]
  (let [new-tokens? (or (:new-tokens? authentication-result)
                        (boolean (not (:tokens authentication-result))))
        tokens (or (:tokens authentication-result)
                   (token-flow/generate-tokens
                    {:user-id user-id
                     :generators (:auth/token-generators request)}))]
    (when new-tokens?
      (proto/create-refresh-token!
       (:auth/auth-store request)
       (-> tokens meta :refresh-claims)))
    (assoc
     (tokens-response request tokens)
     :status (success-status-code request))))

(defn- maximum-cookie-security?
  [cookie-options]
  (and (string? (:path cookie-options))
       (true? (:http-only cookie-options))
       (= :strict (:same-site cookie-options))
       (string? (:domain cookie-options))
       (seq (:domain cookie-options))
       (not= "localhost" (:domain cookie-options))
       (true? (:secure cookie-options))))

(defmethod process-authentication-result :auth/action-denied
  [{:keys [request authentication-result unauthorized-response]}]
  {:events [{:request request
             :type :auth/action-denied
             :action (authentication-result->action authentication-result)}]
   :response (unauthorized-response)})

(defmethod process-authentication-result :auth/handle-authentication-error
  [{:keys [request authentication-result unauthorized-response]}]
  {:events [{:request request
             :type :auth/authentication-error
             :error (:error authentication-result)}]
   :response (cond-> (unauthorized-response)
               (= :flow/validation-error (-> authentication-result :error :type))
               (-> (assoc :body (json/write-str
                                 {:error (:error authentication-result)}
                                 :value-fn (fn [_ v]
                                             (if (keyword? v)
                                               (id/keyword->string v)
                                               v))))
                   (assoc-in [:headers "Content-Type"] "application/json")))})

(defmethod process-authentication-result :auth/handle-authentication-success
  [{:keys [request
           authentication-result
           allow-login?
           unauthorized-response]}]
  (let [allowed? (if (fn? allow-login?)
                   (allow-login? (:user authentication-result))
                   true)]
    {:events [{:request request
               :type (if allowed?
                       :auth/authentication-success
                       :auth/login-denied)}]
     :response (if allowed?
                 (success-tokens-response
                  request authentication-result
                  (-> authentication-result :user :id))
                 (unauthorized-response))}))

(defmethod process-authentication-result :auth/handle-new-user
  [{:keys [request authentication-result
           unauthorized-response
           tx-new-user
           allow-new-user?]}]
  (if (allow-new-user? (:user authentication-result))
    (let [orig-user (:user authentication-result)
          user (tx-new-user orig-user)
          authentication-result (assoc authentication-result :user user)
          sign-up-result (try (sign-up! request authentication-result)
                              (catch Exception e e))]
      (if (throwable? sign-up-result)
        {:events [{:request request
                   :type :auth/sign-up-error
                   :error sign-up-result}]
         :response (unauthorized-response)}
        {:events [{:request request
                   :type :auth/new-user
                   :user sign-up-result
                   :orig-user orig-user}]
         :response (success-tokens-response
                    request authentication-result
                    (:id sign-up-result))}))
    {:events [{:request request
               :user (:user authentication-result)
               :type :auth/disallowed-new-user}]
     :response (unauthorized-response)}))

(defmethod process-authentication-result :default
  [{:keys [request authentication-result unauthorized-response]}]
  {:events [{:request request
             :authentication-result authentication-result
             :type :auth/unknown-authentication-result-case}]
   :response (unauthorized-response)})

(defn- dispatch-events
  [events on-event]
  (doseq [event events] (on-event event)))

(defn- default-event-handler
  [{:keys [request type] :as event}]
  (log/info
   (with-out-str
     (clojure.pprint/pprint
      (compact
       {:re-frame-auth/event
        (merge
         event {:type type
                :request (select-keys
                          request [:headers
                                   :auth/identity
                                   :auth/flow])})})))))

;;; Specs

(s/def :token/generator (partial satisfies? proto/TokenGenerator))
(s/def :token-generator/access-token :token/generator)
(s/def :token-generator/refresh-token :token/generator)
(s/def :token-generator/id-token :token/generator)
(s/def :auth/token-generators
  (s/keys
   :req-un [:token-generator/id-token
            :token-generator/access-token]
   :opt-un [:token-generator/refresh-token]))

(s/def :auth-cookie-option/path string?)
(s/def :auth-cookie-option/http-only boolean?)
(s/def :auth-cookie-option/same-site #{:strict :lax})
(s/def :auth-cookie-option/domain string?)
(s/def :auth-cookie-option/secure boolean?)
(s/def :auth-cookie-option/expires
  #(or (instance? DateTime %)
       (string? %)))

(s/def :auth/cookie-options
  (s/keys :opt-un
          [:auth-cookie-option/path
           :auth-cookie-option/http-only
           :auth-cookie-option/same-site
           :auth-cookie-option/domain
           :auth-cookie-option/secure
           :auth-cookie-option/expires]))

(s/def :auth-action-set/action
  #{:auth/create
    :auth/login})
(s/def :auth/actions-set
  (s/and set? (s/coll-of :auth-action-set/action)))
