(ns hub.cache
  "A simple key-value, time-expiry based in-memory cache."
  (:refer-clojure :exclude [get])
  (:require [clojure.core.cache :as cc]
            [com.stuartsierra.component :as component :refer [Lifecycle]]
            [taoensso.timbre :as log]))

;;; Declarations

;;; Records

(defrecord LRUMemoryCache [threshold]
  Lifecycle
  (start [component]
    (if (:started? component)
      component
      (let [_ (log/debug "starting lru memory cache")
            c (assoc component
                     :cache-atom (atom (cc/lru-cache-factory {} :threshold threshold))
                     :started? true)]
        (log/debug "started lru memory cache")
        c)))
  (stop [component]
    (if-not (:started? component)
      component
      (do (log/debug "stopping cache")
          (let [c (dissoc component
                          :cache-atom
                          :started?)]
            (log/debug "stopped cache")
            c)))))

;;; Public

(defn cache
  ([] (cache 64))
  ([threshold]
   (->LRUMemoryCache threshold)))

(defn get
  [this k]
  (assert (:started? this))
  (let [cache @(:cache-atom this)
        cache* (if (cc/has? cache k) (cc/hit cache k) cache)]
    (reset! (:cache-atom this) cache*)
    (cc/lookup cache* k)))

(defn put!
  [this k v]
  (assert (:started? this))
  (let [cache @(:cache-atom this)]
    (when-not (cc/has? cache k)
      (reset! (:cache-atom this) (cc/miss cache k v)))
    v))

;;; Private
