(ns hub.websocket
  (:require [aleph.http :as http]
            [cheshire.core :as json]
            [clojure.core.async :refer [go chan alts! alts!! >!! <!! <! close! thread]]
            [clojure.string :as st]
            [com.stuartsierra.component :as component]
            [manifold.deferred :as d]
            [manifold.stream :as s]
            [taoensso.timbre :as log])
  (:import [java.net InetSocketAddress]))

;;; Declarations

(def default-port 8080)

(declare server-handler connect)

;;; Records

(defrecord WebsocketServer [host port handler]
  component/Lifecycle
  (start [this]
    (if (:started? this)
      this
      (let [_ (log/debug "starting websocket server")
            cancel-ch (chan)
            c (assoc this
                     :server (http/start-server
                              (server-handler cancel-ch handler)
                              {:socket-address (InetSocketAddress. host port)})
                     :cancel-ch cancel-ch
                     :started? true)]
        (log/debug "started websocket server")
        c)))

  (stop [this]
    (if-not (:started? this)
      this
      (let [{:keys [server cancel-ch]} this]
        (log/debug "stopping websocket server")
        (close! cancel-ch)
        (.close server)
        (let [c (dissoc this :started? :server :cancel-ch)]
          (log/debug "stopped websocket server")
          c)))))

(defrecord WebsocketClient [url handler out-ch]
  component/Lifecycle
  (start [this]
    (if (:started? this)
      this
      (let [cancel-ch (chan)]
        (assoc this
               :client (connect url handler out-ch cancel-ch)
               :cancel-ch cancel-ch
               :started? true))))

  (stop [this]
    (if-not (:started? this)
      this
      (let [{:keys [client cancel-ch]} this]
        (close! cancel-ch)
        (.close client)
        (dissoc this :started? :client :cancel-ch)))))

;;; Public

(defn websocket-server
  ([port handler] (websocket-server "0.0.0.0" port handler))
  ([host port handler]
   (->WebsocketServer host port handler)))

(defn websocket-client
  [url handler out-ch]
  (->WebsocketClient url handler out-ch))

;;; Private

(defn- serialize
  [x]
  (json/generate-string x))

(defn deserialize
  [x]
  (json/parse-string x true))

(defn- consume-websocket
  [s handler {:keys [in-ch out-ch cancel-ch id] :as ws-info}]

  ;; give the custom handler a chance to hook into the websocket
  (try (handler ws-info)
       (catch Exception e
         (log/error e "Error calling websocket on-connect hook.")))

  ;; Register on close
  (s/on-closed s
               (fn []
                 (close! out-ch)
                 (close! in-ch)
                 (log/debug "[Websocket]" "Stream closed.")))

  ;; Process values as they come off the websocket
  (d/loop []
    (d/chain
     (s/take! s ::drained)

     (fn [msg]
       (if (identical? ::drained msg)
         ::drained
         (try (>!! in-ch (deserialize msg))
              (catch Exception e
                (log/error
                 (str "Error occurred reading value from websocket:\n"
                      (with-out-str
                        (clojure.stacktrace/print-cause-trace e))))))))

     (fn [result]
       (if-not (identical? ::drained result)
         (d/recur)
         (log/debug "[Websocket]" "Inbound loop terminated for websocket handler.")))))

  ;; Read messages off the outbound channel, forwarding them down the
  ;; manifold stream to the websocket.
  (go
    (try (loop []
           (let [[v ch] (alts! [out-ch cancel-ch])]
             (when (and v (= ch out-ch))
               (try (s/put! s (serialize v))
                    (catch Exception e
                      (log/error
                       (str "Error occurred writing value to websocket:\n"
                            (with-out-str
                              (clojure.stacktrace/print-cause-trace e))))))
               (recur))))
         (catch Exception e
           (log/error e "Fatal error occurred on websocket connection:\n")))
    (s/close! s)
    (close! out-ch)
    (close! in-ch)
    (log/debug "[Websocket]" "Outbound loop terminated for websocket handler."))

  )

(defn- connect
  [url handler out-ch cancel-ch]
  (let [in-ch (chan)]
    (when-let [s @(http/websocket-client url)]
      (consume-websocket
       s handler
       {:in-ch in-ch
        :out-ch out-ch
        :cancel-ch cancel-ch})
      s)))

(defn- server-handler
  [cancel-ch handler]
  (fn [request]
    (let [out-ch (chan)
          in-ch (chan)
          id (str (java.util.UUID/randomUUID))
          ws-info (assoc request
                         :out-ch out-ch
                         :in-ch in-ch
                         :id id)
          s @(http/websocket-connection request)]
      (consume-websocket s handler (assoc ws-info :cancel-ch cancel-ch)))))
