(ns hub.http.server
  (:require [clojure.core.async :refer [go >! >!! <! chan put!]]
            [clojure.stacktrace :as stack]

            [hub.schema :as hs]

            [schema.core :as s]

            [org.httpkit.server :as http]

            [taoensso.timbre :as log]

            [com.stuartsierra.component :as component :refer [Lifecycle]]))


;;; Declarations

(declare async-handler process-request! start-worker-pool!)

(def ^:private default-port 8080)
(def ^:private default-host "0.0.0.0")
(def ^:private default-workers 100)

;;; Records

(defrecord HttpServer [handler workers host port]
  Lifecycle
  (start [component]
    (if-not (:started component)
      (let [response-worker-ch (chan)]

        ;; Start the pool of workers to process requests
        (start-worker-pool!
         response-worker-ch
         workers
         (partial process-request!
            (dissoc component :handler) handler))

        ;; Return the assembled component
        (assoc component
               :response-worker-ch response-worker-ch
               :stop-server (http/run-server
                             (async-handler
                              response-worker-ch)
                             {:port port
                              :host host})
               :started true))
      component))
  (stop [component]
    (if (:started component)
      (let [{:keys [stop-server response-worker-ch]} component]
        (stop-server)
        (assoc component
               :response-worker-ch nil
               :stop-server nil
               :started false))
      component)))

;;; Public

(defn server
  [handler & {:keys [workers host port]}]
  (map->HttpServer
   {:handler handler
    :workers (or workers default-workers)
    :host (or host default-host)
    :port (or port default-port)}))

;;; Private

(def ^:private error-response {:status 500 :body "An internal error occurred."})

(defn- process-request!
  "Processing a request involves running the given handler on it, and then
  sending a response."
  [component handler [request channel]]
  (when-let [response (or (try (handler (merge component request))
                               (catch Exception e
                                 (log/error request
                                            (with-out-str
                                              (stack/print-cause-trace e)))
                                 nil))
                          error-response)]
    (http/send! channel response)))

(defn- start-worker-pool!
  "Starts a pool of core async workers that read requests from the provided
  channel, and call the provided function on them."
  [ch n f]
  (let [online-workers (atom 0)]
    (dotimes [_ n]
      (go (loop []
            (when-let [v (<! ch)]
              (try (f v)
                   (catch Exception e
                     (log/error e "Error occurred in worker pool.")))
              (recur)))
          (swap! online-workers dec)
          (when (zero? @online-workers)
            (log/info (format "All %s http workers have been brought offline." n))))
      (swap! online-workers inc))
    (log/info (format "All %s http workers have been brought online." n))))

(defn- async-handler
  "Although asynchronous, when a request is received it will block the thread it
  was received on until a request worker accepts the request. This is to prevent
  overloading the workers."
  [response-worker-ch]
  (fn [request]
    (http/with-channel request channel
      (when-not (http/websocket? channel)
        (>!! response-worker-ch [request channel])))))
