(ns hub.cache
  "A simple key-value, time-expiry based in-memory cache."
  (:refer-clojure :exclude [get])
  (:require [clojure.core.cache :as cc]
            [com.stuartsierra.component :as component :refer [Lifecycle]]))

;;; Declarations

;;; Records

(defrecord LRUMemoryCache [threshold]
  Lifecycle
  (start [component]
    (if (:started? component)
      component
      (assoc component
             :cache-atom (atom (cc/lru-cache-factory {} :threshold threshold))
             :started? true)))
  (stop [component]
    (if-not (:started? component)
      component
      (dissoc component
              :cache-atom
              :started?))))

;;; Public

(defn cache
  ([] (cache 64))
  ([threshold]
   (->LRUMemoryCache threshold)))

(defn get
  [this k]
  (assert (:started? this))
  (let [cache @(:cache-atom this)
        cache* (if (cc/has? cache k) (cc/hit cache k) cache)]
    (reset! (:cache-atom this) cache*)
    (cc/lookup cache* k)))

(defn put!
  [this k v]
  (assert (:started? this))
  (let [cache @(:cache-atom this)]
    (when-not (cc/has? cache k)
      (reset! (:cache-atom this) (cc/miss cache k v)))
    v))

;;; Private
