(ns exif-orientation.core
  (:require [utilis.map :refer [compact]]
            [clj-exif.core :as exif]
            [clojure.java.io :refer [file input-stream output-stream]]
            [clojure.java.io :as io]
            [clojure.string :as st])
  (:import [java.io InputStream File ByteArrayInputStream
            FileInputStream FileOutputStream
            IOException ByteArrayOutputStream]
           [org.apache.commons.imaging Imaging]
           [javax.imageio ImageIO]
           [java.awt Color]
           [java.awt.image BufferedImage BufferedImageOp]
           [org.apache.commons.io IOUtils]
           [org.imgscalr Scalr Scalr$Rotation]))

;;; Declarations

(declare decode upright image->byte-array)

(def ^:private output-formats #{:jpeg :png})

;;; Public

(defn strip-exif-metadata
  "The argument 'image' can be a file, path to a file, or byte array."
  [image & {:keys [output-format] :or {output-format :jpeg}}]
  (when-not (output-formats output-format)
    (throw
     (ex-info (str "Unrecognized output format " output-format)
              {:output-format output-format
               :allowed-formats output-formats})))
  (when-let [is (input-stream (if (string? image) (io/file image) image))]
    (-> is decode upright (image->byte-array output-format))))

(defn write-file
  [filename b]
  (with-open [w (java.io.BufferedOutputStream.
                 (java.io.FileOutputStream. filename))]
    (.write w b))
  (io/file filename))

;;; Private

(defn- sanitize-output-format
  [output-format]
  (st/lower-case
   (if (string? output-format)
     output-format
     ({:png "png" :jpeg "jpg"} output-format))))

(defn- convert-image
  [input-image input-format output-format]
  (if (= input-format output-format)
    input-image
    (if-let [new-image (BufferedImage.
                        (.getWidth input-image)
                        (.getHeight input-image)
                        (condp = output-format
                          "jpg" (BufferedImage/TYPE_INT_RGB)
                          "png" (BufferedImage/TYPE_INT_ARGB)
                          nil))]
      (do (-> new-image
              (.createGraphics)
              (.drawImage input-image 0 0 Color/BLACK nil))
          new-image)
      (throw
       (ex-info
        "Unable to convert image."
        {:input-image input-image
         :input-format input-format
         :output-format output-format})))))

(defn- image->byte-array
  ([image] (image->byte-array image (:format image)))
  ([image output-format]
   (let [output-format (sanitize-output-format output-format)
         image (convert-image (:image image) (:format image) output-format)
         baos (ByteArrayOutputStream.)]
     (ImageIO/write image output-format baos)
     (.toByteArray baos))))

(defn- image-format
  [is]
  (let [readers (ImageIO/getImageReaders
                 (ImageIO/createImageInputStream is))]
    (when-let [reader (and (.hasNext readers) (.next readers))]
      (.getFormatName reader))))

(defn- metadata
  [b]
  (Imaging/getMetadata b))

(defn- decode
  [is]
  (let [b (IOUtils/toByteArray is)]
    (compact
     {:image (ImageIO/read (input-stream b))
      :bytes b
      :format (image-format (input-stream b))
      :metadata (metadata b)})))

(defn- rotate
  [^BufferedImage image ^String rotation]
  (Scalr/rotate
   image (Scalr$Rotation/valueOf rotation)
   (into-array java.awt.image.BufferedImageOp [])))

(defn- orientation->rotations
  [orientation]
  (case orientation
    2 ["FLIP_HORZ"]
    3 ["CW_180"]
    4 ["FLIP_VERT"]
    5 ["FLIP_VERT" "CW_90"]
    6 ["CW_90"]
    7 ["FLIP_HORZ" "CW_90"]
    8 ["CW_270"]
    nil))

(defn- rotations
  [metadata]
  (when (and metadata (.getExif metadata))
    (orientation->rotations
     (get-in (exif/read metadata)
             ["Root" "Orientation"]))))

(defn- upright
  [decoded-image]
  (update decoded-image :image
          #(or (->> (:metadata decoded-image)
                    rotations (reduce rotate %))
               %) ))
