(ns escpos.core
  (:refer-clojure :exclude [newline])
  (:require [utilis.js :as j]
            [clojure.string :as st]))

;;; Declarations

(declare queue wrap-line encode* state->number)

;;; API

(defn encoder
  []
  {:buffer []
   :codepage :ascii
   :bold :off
   :italic :off
   :underline :off ;; :on/:off/:double
   :hanzi :off})

(defn initialize
  [encoder]
  (queue encoder [0x1b 0x40]))

(defn text
  ([encoder value] (text encoder value nil))
  ([encoder value wrap]
   (let [value (if wrap
                 (wrap-line value
                            :line-break "\r\n"
                            :size wrap)
                 value)
         bytes (encode* encoder value)]
     (if (not= :off (:hanzi encoder))
       (queue encoder [0x1c 0x26 bytes 0x1c 0x2e])
       (queue encoder [bytes])))))

(defn newline
  [encoder]
  (queue encoder [0x0a 0x0d]))

(defn line
  ([encoder value] (line encoder value nil))
  ([encoder value wrap]
   (-> encoder
       (text value wrap)
       (newline))))

(defn underline
  [encoder value]
  (-> encoder
      (assoc :underline value)
      (queue [0x1b 0x2d (state->number value)])))

(defn italic
  [encoder value]
  (-> encoder
      (assoc :italic value)
      (queue [0x1b 0x34 (state->number value)])))

(defn bold
  [encoder value]
  (-> encoder
      (assoc :bold value)
      (queue [0x1b 0x45 (state->number value)])))

(defn size
  [encoder value]
  (queue encoder
         [0x1b 0x4d
          (if (= value :small)
            0x01
            0x00)]))

(defn align
  [encoder value]
  (let [alignments {:left 0x00 :center 0x01 :right 0x02}]
    (queue encoder [0x1b 0x61 (alignments value)])))

(defn barcode
  [encoder value symbology height]
  (let [symbologies {:upca 0x00
                     :upce 0x01
                     :ean13 0x02
                     :ean8 0x03
                     :coda39 0x04
                     :itf 0x05
                     :codabar 0x06}
        bytes (encode* (assoc encoder :codepage :ascii) value)]
    (queue encoder [0x1d 0x68 height
                    0x1d 0x77 (if (= symbology :code39) 0x02 0x03)
                    0x1d 0x6b (symbologies symbology)
                    bytes
                    0x00])))

(defn qrcode
  ([encoder value] (qrcode encoder value 2 6 :m))
  ([encoder value model size errorlevel]

   ;; https://github.com/NielsLeenheer/EscPosEncoder/blob/master/src/esc-pos-encoder.js#L357

   encoder)

  )

(defn image
  [encoder element width height algorithm threshold]

  ;; https://github.com/NielsLeenheer/EscPosEncoder/blob/master/src/esc-pos-encoder.js#L451

  encoder

  )

(defn cut
  [encoder]
  (queue encoder [0x1b 0x69]))

(defn raw
  [encoder data]
  (queue encoder data))

(defn encode
  [encoder]
  (flatten (:buffer encoder)))

;;; Private

(defn- queue
  [encoder value]
  (reduce (fn [encoder value]
            (update encoder :buffer conj value))
          encoder
          value))

(defn- wrap-line
  [text & {:keys [line-break size]
           :or {line-break "\n"
                size 40}}]
  (let [lines (loop [left size line [] lines []
                     words (st/split text #"\s+")]
                (if-let [word (first words)]
                  (let [wlen (count word)
                        spacing (if (== left size) "" " ")
                        alen (+ (count spacing) wlen)]
                    (if (<= alen left)
                      (recur (- left alen) (conj line spacing word) lines (next words))
                      (recur (- size wlen) [word] (conj lines (apply str line)) (next words))))
                  (when (seq line)
                    (conj lines (apply str line)))))]
    (st/join line-break lines)))

(defn- encode*
  [encoder value]
  (condp = (:codepage encoder)
    :ascii (mapv #(j/call % :charCodeAt 0) value)))

(defn- state->number
  [state]
  ({:off 0
    :on 1
    :double 2} state))
